<?php
/**
 * Plugin Name: Digital Asset Hub
 * Description: Spiel-Loader mit Alias-URL → echter versteckter Ordner (Lösung 2). ZIP-Import, FTP-Scanner, Alias-Mapping und direkte Dateiserver-Unterstützung unter /play/{alias}/ so dass relative Pfade intakt bleiben.
 * Version:     1.1.0
 * Author: Fensal
 * Text Domain: zocker-loge.de
 *
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Direkten Zugriff verhindern
}

// ===============================================
// 1. REGISTRIERUNG VON CUSTOM POST TYPE UND TAXONOMIE
// ===============================================

function slgs_register_game_post_type() {
    $labels = array(
        'name'                  => _x( 'Spiele', 'Post Type General Name', 'steam-like-shop' ),
        'singular_name'         => _x( 'Spiel', 'Post Type Singular Name', 'steam-like-shop' ),
        'menu_name'             => __( 'Spiele-Shop', 'steam-like-shop' ),
        'all_items'             => __( 'Alle Spiele', 'steam-like-shop' ),
        'add_new_item'          => __( 'Neues Spiel hinzufügen', 'steam-like-shop' ),
        'add_new'               => __( 'Neu hinzufügen', 'steam-like-shop' ),
        'edit_item'             => __( 'Spiel bearbeiten', 'steam-like-shop' ),
        'update_item'           => __( 'Spiel aktualisieren', 'steam-like-shop' ),
        'view_item'             => __( 'Spiel ansehen', 'steam-like-shop' ),
        'view_items'            => __( 'Spiele ansehen', 'steam-like-shop' ),
    );
    $args = array(
        'label'                 => __( 'Spiel', 'steam-like-shop' ),
        'labels'                => $labels,
        'supports'              => array( 'title', 'editor', 'thumbnail', 'excerpt' ),
        'hierarchical'          => false,
        'public'                => true,
        'show_ui'               => true,
        'show_in_menu'          => true,
        'menu_position'         => 5,
        'menu_icon'             => 'dashicons-games',
        'has_archive'           => 'games',
        'publicly_queryable'    => true,
        'capability_type'       => 'post',
        'show_in_rest'          => true,
    );
    register_post_type( 'game', $args );

    // Custom Taxonomy: Spiel-Kategorien
    $labels_cat = array(
        'name'                       => _x( 'Spiel-Kategorien', 'Taxonomy General Name', 'steam-like-shop' ),
        'singular_name'              => _x( 'Spiel-Kategorie', 'Taxonomy Singular Name', 'steam-like-shop' ),
        'menu_name'                  => __( 'Kategorien', 'steam-like-shop' ),
        'all_items'                  => __( 'Alle Kategorien', 'steam-like-shop' ),
        'new_item_name'              => __( 'Name der neuen Kategorie', 'steam-like-shop' ),
        'add_new_item'               => __( 'Neue Kategorie hinzufügen', 'steam-like-shop' ),
        'edit_item'                  => __( 'Kategorie bearbeiten', 'steam-like-shop' ),
        'update_item'                => __( 'Kategorie aktualisieren', 'steam-like-shop' ),
    );
    $args_cat = array(
        'labels'                     => $labels_cat,
        'hierarchical'               => true,
        'public'                     => true,
        'show_ui'                    => true,
        'show_admin_column'          => true,
        'show_in_nav_menus'          => true,
        'show_tagcloud'              => false,
        'show_in_rest'               => true,
    );
    register_taxonomy( 'game_category', array( 'game' ), $args_cat );
}
add_action( 'init', 'slgs_register_game_post_type', 0 );


// ===============================================
// 1.5 ADMIN MENUS (Shortcodes & Einstellungen)
// ===============================================

function slgs_add_admin_pages() {
    // Shortcode-Seite (Untermenü)
    add_submenu_page(
        'edit.php?post_type=game',
        __( 'Shortcode-Übersicht', 'steam-like-shop' ),
        __( 'Shortcodes', 'steam-like-shop' ),
        'manage_options',
        'slgs-shortcodes',
        'slgs_shortcode_page_content'
    );

    // Einstellungsseite (Untermenü)
    add_submenu_page(
        'edit.php?post_type=game',
        __( 'Einstellungen', 'steam-like-shop' ),
        __( 'Einstellungen', 'steam-like-shop' ),
        'manage_options',
        'slgs-settings',
        'slgs_settings_page_content'
    );
}
add_action( 'admin_menu', 'slgs_add_admin_pages' );

/**
 * Inhalt der Shortcode-Übersichtsseite
 */
function slgs_shortcode_page_content() {
    ?>
    <div class="wrap">
        <h1><?php _e( 'Shortcode-Übersicht für den Steam-ähnlichen Spiele-Shop', 'steam-like-shop' ); ?></h1>
        <p class="description">
            <?php _e( 'Verwenden Sie die folgenden Shortcodes, um die Spiele-Listen, den Warenkorb und den zentralen Hub auf jeder Ihrer WordPress-Seiten oder in einem Widget anzuzeigen.', 'steam-like-shop' ); ?>
        </p>

        <style>
            .slgs-shortcode-table { width: 100%; border-collapse: collapse; margin-top: 20px; }
            .slgs-shortcode-table th, .slgs-shortcode-table td { padding: 12px; border: 1px solid #ddd; text-align: left; }
            .slgs-shortcode-table th { background-color: #f3f3f3; font-weight: bold; width: 25%; }
            .slgs-shortcode { font-family: monospace; font-size: 1.1em; background-color: #ffe; padding: 2px 5px; border: 1px dashed #ccc; border-radius: 3px; user-select: all; }
        </style>

        <table class="slgs-shortcode-table">
            <thead>
                <tr>
                    <th><?php _e( 'Shortcode', 'steam-like-shop' ); ?></th>
                    <th><?php _e( 'Beschreibung', 'steam-like-shop' ); ?></th>
                    <th><?php _e( 'Hinweis', 'steam-like-shop' ); ?></th>
                </tr>
            </thead>
            <tbody>
                <tr><td><span class="slgs-shortcode">[slgs_main_hub]</span></td><td><?php _e( 'Der zentrale Hub mit Tabs für Store, Neu, Meistgespielt, Top-Bewertet, Bibliothek und Wunschliste.', 'steam-like-shop' ); ?></td><td><?php _e( 'Wird für die Hauptseite des Shops empfohlen. **Der Store-Tab enthält nun das Kategorie-Dropdown.**', 'steam-like-shop' ); ?></td></tr>
                <tr><td><span class="slgs-shortcode">[game_shop]</span></td><td><?php _e( 'Zeigt die vollständige Liste aller Spiele an.', 'steam-like-shop' ); ?></td><td><?php _e( 'Dasselbe wie der "Store"-Tab im Hub (ohne Filterleiste).', 'steam-like-shop' ); ?></td></tr>
                <tr><td><span class="slgs-shortcode">[slgs_cart_checkout]</span></td><td><?php _e( 'Zeigt den Warenkorb-Inhalt und den Checkout-Button für kostenlose Spiele an.', 'steam-like-shop' ); ?></td><td><?php _e( 'Sollte auf einer dedizierten Warenkorb-Seite platziert werden. Erfordert Anmeldung.', 'steam-like-shop' ); ?></td></tr>
                <tr><td><span class="slgs-shortcode">[new_games]</span></td><td><?php _e( 'Zeigt die 6 neuesten Spiele.', 'steam-like-shop' ); ?></td><td><?php _e( 'Ideal für eine Sektion "Neuheiten".', 'steam-like-shop' ); ?></td></tr>
                <tr><td><td><span class="slgs-shortcode">[top_games]</span></td></td><td><?php _e( 'Zeigt die 6 meistgespielten Spiele (nach Aktivierungen).', 'steam-like-shop' ); ?></td><td><?php _e( 'Ideal für eine Sektion "Bestseller".', 'steam-like-shop' ); ?></td></tr>
                <tr><td><td><span class="slgs-shortcode">[best_rated_games]</span></td></td><td><?php _e( 'Zeigt die 6 am besten bewerteten Spiele.', 'steam-like-shop' ); ?></td><td><?php _e( 'Ideal für eine Sektion "Top-Bewertungen".', 'steam-like-shop' ); ?></td></tr>
                <tr><td><span class="slgs-shortcode">[my_games]</span></td><td><?php _e( 'Zeigt die Spiele in der Benutzerbibliothek.', 'steam-like-shop' ); ?></td><td><?php _e( 'Erfordert eine Benutzeranmeldung.', 'steam-like-shop' ); ?></td></tr>
                <tr><td><span class="slgs-shortcode">[my_wishlist]</span></td><td><?php _e( 'Zeigt die Spiele auf der Benutzer-Wunschliste.', 'steam-like-shop' ); ?></td><td><?php _e( 'Erfordert eine Benutzeranmeldung.', 'steam-like-shop' ); ?></td></tr>
                <tr><td><span class="slgs-shortcode">[slgs_cart_button]</span></td><td><?php _e( 'Zeigt das Warenkorb-Widget (Icon und Zähler) an.', 'steam-like-shop' ); ?></td><td><?php _e( 'Wird normalerweise im Header oder in der Sidebar verwendet.', 'steam-like-shop' ); ?></td></tr>
                <tr><td><span class="slgs-shortcode">[latest_reviews]</span></td><td><?php _e( 'Platzhalter für Community-Highlights.', 'steam-like-shop' ); ?></td><td><?php _e( '(Text-Rezensionen wurden entfernt.)', 'steam-like-shop' ); ?></td></tr>
            </tbody>
        </table>
    </div>
    <?php
}

/**
 * Registriert die Einstellungen
 */
function slgs_register_settings() {
    register_setting( 'slgs_settings_group', 'slgs_shop_title', array(
        'type' => 'string',
        'sanitize_callback' => 'sanitize_text_field',
        'default' => __( 'Game Shop Übersicht', 'steam-like-shop' ),
    ) );
    register_setting( 'slgs_settings_group', 'slgs_currency_symbol', array(
        'type' => 'string',
        'sanitize_callback' => 'sanitize_text_field',
        'default' => '€',
    ) );
    register_setting( 'slgs_settings_group', 'slgs_checkout_page_slug', array(
        'type' => 'string',
        'sanitize_callback' => 'sanitize_title',
        'default' => 'checkout',
    ) );
    // Einstellung für die "Einkauf fortsetzen" Seite
     register_setting( 'slgs_settings_group', 'slgs_continue_shopping_url', array(
        'type' => 'string',
        'sanitize_callback' => 'esc_url_raw',
        'default' => get_post_type_archive_link( 'game' ),
    ) );
    // NEU: Slug/ID der Haupt-Shop-Seite, auf der [slgs_main_hub] liegt.
    register_setting( 'slgs_settings_group', 'slgs_main_shop_url', array(
        'type' => 'string',
        'sanitize_callback' => 'esc_url_raw',
        'default' => home_url( '/' ),
    ) );
}
add_action( 'admin_init', 'slgs_register_settings' );

/**
 * Inhalt der Einstellungsseite
 */
function slgs_settings_page_content() {
    $currency_symbol = get_option( 'slgs_currency_symbol', '€' );
    $checkout_slug = get_option( 'slgs_checkout_page_slug', 'checkout' );
    $continue_shopping_url = get_option( 'slgs_continue_shopping_url', get_post_type_archive_link( 'game' ) );
    $main_shop_url = get_option( 'slgs_main_shop_url', home_url( '/' ) );
    ?>
    <div class="wrap">
        <h1><?php _e( 'Shop-Einstellungen', 'steam-like-shop' ); ?></h1>
        <form method="post" action="options.php">
            <?php settings_fields( 'slgs_settings_group' ); ?>
            <?php do_settings_sections( 'slgs_settings_group' ); ?>

            <table class="form-table">
                <tr valign="top">
                    <th scope="row"><label for="slgs_shop_title"><?php _e('Titel der Hauptübersicht', 'steam-like-shop'); ?></label></th>
                    <td>
                        <input type="text" id="slgs_shop_title" name="slgs_shop_title" value="<?php echo esc_attr( get_option( 'slgs_shop_title', __( 'Game Shop Übersicht', 'steam-like-shop' ) ) ); ?>" class="regular-text" />
                        <p class="description"><?php _e('Dieser Titel wird oben im [slgs_main_hub] Shortcode angezeigt.', 'steam-like-shop'); ?></p>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row"><label for="slgs_currency_symbol"><?php _e('Währungssymbol', 'steam-like-shop'); ?></label></th>
                    <td>
                        <input type="text" id="slgs_currency_symbol" name="slgs_currency_symbol" value="<?php echo esc_attr( $currency_symbol ); ?>" class="small-text" />
                        <p class="description"><?php _e('Geben Sie das Währungssymbol ein (z.B. €, $, £).', 'steam-like-shop'); ?></p>
                    </td>
                </tr>
                 <tr valign="top">
                    <th scope="row"><label for="slgs_checkout_page_slug"><?php _e('Checkout Seiten-Slug', 'steam-like-shop'); ?></label></th>
                    <td>
                        <input type="text" id="slgs_checkout_page_slug" name="slgs_checkout_page_slug" value="<?php echo esc_attr( $checkout_slug ); ?>" class="regular-text" />
                        <p class="description"><?php _e('Geben Sie den Slug der Seite ein, auf der Sie den Shortcode [slgs_cart_checkout] platziert haben (z.B. "warenkorb").', 'steam-like-shop'); ?></p>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row"><label for="slgs_continue_shopping_url"><?php _e('URL "Einkauf fortsetzen"', 'steam-like-shop'); ?></label></th>
                    <td>
                        <input type="url" id="slgs_continue_shopping_url" name="slgs_continue_shopping_url" value="<?php echo esc_attr( $continue_shopping_url ); ?>" class="regular-text" />
                        <p class="description"><?php _e('Die URL, zu der der Button "Einkauf fortsetzen" auf der Warenkorbseite verlinkt (z.B. die Archivseite Ihrer Spiele).', 'steam-like-shop'); ?></p>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row"><label for="slgs_main_shop_url"><?php _e('Haupt-Shop-Seite URL (mit [slgs_main_hub])', 'steam-like-shop'); ?></label></th>
                    <td>
                        <input type="url" id="slgs_main_shop_url" name="slgs_main_shop_url" value="<?php echo esc_attr( $main_shop_url ); ?>" class="regular-text" />
                        <p class="description"><?php _e('Die URL der Seite, auf der Sie den Shortcode [slgs_main_hub] platziert haben. Wichtig für die korrekte Navigation der Menüleiste.', 'steam-like-shop'); ?></p>
                    </td>
                </tr>
            </table>

            <?php submit_button(); ?>
        </form>
    </div>
    <?php
}


// ===============================================
// 1.6 FULL-WIDTH BODY CLASS FÜR SPIEL-SEITEN
// ===============================================

function slgs_add_full_width_body_class( $classes ) {
    if ( is_singular( 'game' ) || is_post_type_archive( 'game' ) || is_tax( 'game_category' ) ) {
        $classes[] = 'slgs-full-width';
    }
    return $classes;
}
add_filter( 'body_class', 'slgs_add_full_width_body_class' );


// ===============================================
// 2. METAFELDER (Zusätzliche Spieldetails - ERWEITERT)
// ===============================================

function slgs_add_meta_box() {
    add_meta_box(
        'slgs_game_details',
        __( 'Spieldetails & Media', 'steam-like-shop' ),
        'slgs_meta_box_callback',
        'game',
        'normal',
        'high'
    );
}
add_action( 'add_meta_boxes', 'slgs_add_meta_box' );

function slgs_meta_box_callback( $post ) {
    wp_nonce_field( 'slgs_save_meta_box_data', 'slgs_meta_box_nonce' );

    $creator         = get_post_meta( $post->ID, '_slgs_creator', true );
    $trailer_url     = get_post_meta( $post->ID, '_slgs_trailer_url', true );
    $game_url        = get_post_meta( $post->ID, '_slgs_game_url', true );
    $download_url    = get_post_meta( $post->ID, '_slgs_download_url', true ); // HINZUGEFÜGT
    $screenshot_urls = get_post_meta( $post->ID, '_slgs_screenshot_urls', true ); // Zeilenumbrüche
    $screenshot_array = array_filter( array_map( 'trim', explode( "\n", $screenshot_urls ) ) );

    $activation_count = get_post_meta( $post->ID, '_slgs_activation_count', true );
    $average_rating  = get_post_meta( $post->ID, '_slgs_average_rating', true );
    $total_ratings   = get_post_meta( $post->ID, '_slgs_total_ratings', true );
    $rating_sum      = get_post_meta( $post->ID, '_slgs_rating_sum', true );

    $is_multiplayer  = get_post_meta( $post->ID, '_slgs_is_multiplayer', true );
    $is_singleplayer = get_post_meta( $post->ID, '_slgs_is_singleplayer', true );
    $is_coop         = get_post_meta( $post->ID, '_slgs_is_coop', true );
    $fsk_rating      = get_post_meta( $post->ID, '_slgs_fsk_rating', true );
    $languages       = get_post_meta( $post->ID, '_slgs_supported_languages', true );

    $price           = get_post_meta( $post->ID, '_slgs_price', true );
    $min_os          = get_post_meta( $post->ID, '_slgs_min_os', true );
    $min_cpu         = get_post_meta( $post->ID, '_slgs_min_cpu', true );
    $min_ram         = get_post_meta( $post->ID, '_slgs_min_ram', true );
    $min_gpu         = get_post_meta( $post->ID, '_slgs_min_gpu', true );
    $rec_os          = get_post_meta( $post->ID, '_slgs_rec_os', true );
    $rec_cpu         = get_post_meta( $post->ID, '_slgs_rec_cpu', true );
    $rec_ram         = get_post_meta( $post->ID, '_slgs_rec_ram', true );
    $rec_gpu         = get_post_meta( $post->ID, '_slgs_rec_gpu', true );

    $update_log      = get_post_meta( $post->ID, '_slgs_update_log', true );
    ?>

    <h3><?php _e('Basisinformationen', 'steam-like-shop'); ?></h3>
    <p><label for="slgs_creator"><?php _e( 'Ersteller/Entwickler:', 'steam-like-shop' ); ?></label><input type="text" id="slgs_creator" name="slgs_creator" value="<?php echo esc_attr( $creator ); ?>" class="widefat" /></p>
    <p><label for="slgs_game_url"><?php _e( 'Spiel-URL (Link zum Browser Game):', 'steam-like-shop' ); ?></label><input type="url" id="slgs_game_url" name="slgs_game_url" value="<?php echo esc_attr( $game_url ); ?>" class="widefat" required /><small class="description"><?php _e( 'Dies ist der Link, der beim Klick auf "Spielen" verwendet wird.', 'steam-like-shop' ); ?></small></p>
    <p><label for="slgs_download_url"><?php _e( 'Download-URL (Optionale Datei/Installer):', 'steam-like-shop' ); ?></label><input type="url" id="slgs_download_url" name="slgs_download_url" value="<?php echo esc_attr( $download_url ); ?>" class="widefat" /><small class="description"><?php _e( 'Optionaler Link für einen direkten Download. Wird nur in der Bibliothek angezeigt.', 'steam-like-shop' ); ?></small></p>

    <h3><?php _e('Preisgestaltung & Systemanforderungen', 'steam-like-shop'); ?></h3>
    <table class="form-table">
        <tr>
            <th scope="row"><label for="slgs_price"><?php _e('Preis (Euro):', 'steam-like-shop'); ?></label></th>
            <td><input type="number" step="0.01" min="0" id="slgs_price" name="slgs_price" value="<?php echo esc_attr( $price ); ?>" class="regular-text" /><small class="description"><?php _e( '0.00 für kostenlos.', 'steam-like-shop'); ?></small></td>
        </tr>
    </table>

    <div style="display: flex; gap: 20px;">
        <div style="flex: 1;">
            <h4><?php _e('Minimale Systemanforderungen', 'steam-like-shop'); ?></h4>
            <p><label for="slgs_min_os"><?php _e('Betriebssystem (OS):', 'steam-like-shop'); ?></label><input type="text" id="slgs_min_os" name="slgs_min_os" value="<?php echo esc_attr( $min_os ); ?>" class="widefat" /></p>
            <p><label for="slgs_min_cpu"><?php _e('Prozessor (CPU):', 'steam-like-shop'); ?></label><input type="text" id="slgs_min_cpu" name="slgs_min_cpu" value="<?php echo esc_attr( $min_cpu ); ?>" class="widefat" /></p>
            <p><label for="slgs_min_ram"><?php _e('Arbeitsspeicher (RAM):', 'steam-like-shop'); ?></label><input type="text" id="slgs_min_ram" name="slgs_min_ram" value="<?php echo esc_attr( $min_ram ); ?>" class="widefat" /></p>
            <p><label for="slgs_min_gpu"><?php _e('Grafikkarte (GPU):', 'steam-like-shop'); ?></label><input type="text" id="slgs_min_gpu" name="slgs_min_gpu" value="<?php echo esc_attr( $min_gpu ); ?>" class="widefat" /></p>
        </div>
        <div style="flex: 1;">
            <h4><?php _e('Empfohlene Systemanforderungen', 'steam-like-shop'); ?></h4>
            <p><label for="slgs_rec_os"><?php _e('Betriebssystem (OS):', 'steam-like-shop'); ?></label><input type="text" id="slgs_rec_os" name="slgs_rec_os" value="<?php echo esc_attr( $rec_os ); ?>" class="widefat" /></p>
            <p><label for="slgs_rec_cpu"><?php _e('Prozessor (CPU):', 'steam-like-shop'); ?></label><input type="text" id="slgs_rec_cpu" name="slgs_rec_cpu" value="<?php echo esc_attr( $rec_cpu ); ?>" class="widefat" /></p>
            <p><label for="slgs_rec_ram"><?php _e('Arbeitsspeicher (RAM):', 'steam-like-shop'); ?></label><input type="text" id="slgs_rec_ram" name="slgs_rec_ram" value="<?php echo esc_attr( $rec_ram ); ?>" class="widefat" /></p>
            <p><label for="slgs_rec_gpu"><?php _e('Grafikkarte (GPU):', 'steam-like-shop'); ?></label><input type="text" id="slgs_rec_gpu" name="slgs_rec_gpu" value="<?php echo esc_attr( $rec_gpu ); ?>" class="widefat" /></p>
        </div>
    </div>

    <h3><?php _e('Spiel-Features (Frontend Icons)', 'steam-like-shop'); ?></h3>
    <table class="form-table">
        <tr>
            <th scope="row"><?php _e('Gameplay Typ:', 'steam-like-shop'); ?></th>
            <td>
                <label><input type="checkbox" name="slgs_is_singleplayer" value="1" <?php checked( $is_singleplayer, 1 ); ?> /><?php _e('Einzelspieler (Singleplayer)', 'steam-like-shop'); ?></label><br>
                <label><input type="checkbox" name="slgs_is_multiplayer" value="1" <?php checked( $is_multiplayer, 1 ); ?> /><?php _e('Mehrspieler (Multiplayer)', 'steam-like-shop'); ?></label><br>
                <label><input type="checkbox" name="slgs_is_coop" value="1" <?php checked( $is_coop, 1 ); ?> /><?php _e('Kooperativ (Co-Op)', 'steam-like-shop'); ?></label>
            </td>
        </tr>
        <tr>
            <th scope="row"><label for="slgs_fsk_rating"><?php _e('Altersfreigabe (FSK):', 'steam-like-shop'); ?></label></th>
            <td>
                <select id="slgs_fsk_rating" name="slgs_fsk_rating">
                    <option value="" <?php selected( $fsk_rating, '' ); ?>>-- <?php _e('Nicht angegeben', 'steam-like-shop'); ?> --</option>
                    <?php
                    $fsk_options = array('FSK 0', 'FSK 6', 'FSK 12', 'FSK 16', 'FSK 18');
                    foreach ($fsk_options as $fsk) {
                        echo '<option value="' . esc_attr($fsk) . '" ' . selected( $fsk_rating, $fsk, false ) . '>' . esc_html($fsk) . '</option>';
                    }
                    ?>
                </select>
            </td>
        </tr>
        <tr>
            <th scope="row"><label for="slgs_supported_languages"><?php _e('Unterstützte Sprachen:', 'steam-like-shop'); ?></label></th>
            <td><input type="text" id="slgs_supported_languages" name="slgs_supported_languages" value="<?php echo esc_attr( $languages ); ?>" class="widefat" /><small class="description"><?php _e( 'Kommagetrennt eingeben (z.B.: Deutsch, Englisch, Französisch).', 'steam-like-shop'); ?></small></td>
        </tr>
    </table>

    <h3><?php _e('Media (Trailer & Screenshots)', 'steam-like-shop'); ?></h3>
    <p><label for="slgs_trailer_url"><?php _e( 'Trailer-URL (z.B. YouTube-Link):', 'steam-like-shop' ); ?></label><input type="url" id="slgs_trailer_url" name="slgs_trailer_url" value="<?php echo esc_attr( $trailer_url ); ?>" class="widefat" /><small class="description"><?php _e( 'Wird auf der Detailseite eingebettet.', 'steam-like-shop'); ?></small></p>

    <div class="slgs-screenshots-container">
        <p><label><?php _e( 'Screenshots aus der Medienbibliothek:', 'steam-like-shop' ); ?></label></p>
        <input type="hidden" id="slgs_screenshot_urls" name="slgs_screenshot_urls" value="<?php echo esc_attr( $screenshot_urls ); ?>" class="widefat" />
        <button type="button" class="button button-secondary" id="slgs-upload-screenshots"><?php _e('Bilder auswählen/hinzufügen', 'steam-like-shop'); ?></button>
        <button type="button" class="button button-secondary" id="slgs-clear-screenshots" <?php echo empty($screenshot_urls) ? 'style="display:none;"' : ''; ?>><?php _e('Bilder löschen', 'steam-like-shop'); ?></button>
        <p class="description"><?php _e( 'Wählen Sie mehrere Bilder für die Galerie aus. Die URLs werden hier gespeichert (eine pro Zeile). Das Titelbild (Featured Image) wird automatisch als erstes Bild in der Galerie verwendet.', 'steam-like-shop'); ?></p>

        <div id="slgs-screenshots-preview" style="margin-top: 10px; display: flex; flex-wrap: wrap; gap: 10px; border: 1px solid #ccc; padding: 10px; border-radius: 4px; background: #f9f9f9; min-height: 50px;">
            <?php foreach ($screenshot_array as $url): ?>
                <img src="<?php echo esc_url($url); ?>" style="width: 80px; height: 80px; object-fit: cover; border-radius: 4px;" />
            <?php endforeach; ?>
        </div>
    </div>


    <h3><?php _e('Updates (Changelog)', 'steam-like-shop'); ?></h3>
    <p><label for="slgs_update_log"><?php _e( 'Versions- und Update-Log:', 'steam-like-shop'); ?></label><textarea id="slgs_update_log" name="slgs_update_log" class="widefat" rows="10"><?php echo esc_textarea( $update_log ); ?></textarea><small class="description"><?php _e( 'Geben Sie Versionsnummern und Änderungen ein (eine Zeile pro Update).', 'steam-like-shop'); ?></small></p>

    <h3><?php _e('Statistiken (Nur Anzeige)', 'steam-like-shop'); ?></h3>
    <p><strong><?php _e( 'Gesamte Aktivierungen (Klicks auf Spielen):', 'steam-like-shop'); ?></strong><span id="slgs_activation_count_display"><?php echo intval($activation_count); ?></span><input type="hidden" name="_slgs_activation_count" value="<?php echo intval($activation_count); ?>"></p>
    <p><strong><?php _e( 'Gesamtpunktzahl (Rating Summe):', 'steam-like-shop'); ?></strong><span id="slgs_rating_sum_display"><?php echo intval($rating_sum); ?></span></p>
    <p><strong><?php _e( 'Anzahl der Bewertungen (Total Votes):', 'steam-like-shop'); ?></strong><span id="slgs_total_ratings_display"><?php echo intval($total_ratings); ?></span></p>
    <p><strong><?php _e( 'Durchschnittliche Bewertung (Anzeige):', 'steam-like-shop'); ?></strong><?php echo number_format((float)$average_rating, 1, ',', ''); ?> / 5.0</p>
    <?php
}

function slgs_save_meta_box_data( $post_id ) {
    if ( ! isset( $_POST['slgs_meta_box_nonce'] ) || ! wp_verify_nonce( $_POST['slgs_meta_box_nonce'], 'slgs_save_meta_box_data' ) ) { return; }
    if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) { return; }
    if ( ! current_user_can( 'edit_post', $post_id ) ) { return; }

    if ( isset( $_POST['slgs_creator'] ) ) { update_post_meta( $post_id, '_slgs_creator', sanitize_text_field( $_POST['slgs_creator'] ) ); }
    if ( isset( $_POST['slgs_trailer_url'] ) ) { update_post_meta( $post_id, '_slgs_trailer_url', esc_url_raw( $_POST['slgs_trailer_url'] ) ); }
    if ( isset( $_POST['slgs_game_url'] ) ) { update_post_meta( $post_id, '_slgs_game_url', esc_url_raw( $_POST['slgs_game_url'] ) ); }
    if ( isset( $_POST['slgs_download_url'] ) ) { update_post_meta( $post_id, '_slgs_download_url', esc_url_raw( $_POST['slgs_download_url'] ) ); } // HINZUGEFÜGT
    // Speichert nun die Zeilenumbrüche der URLs aus dem Hidden-Feld
    if ( isset( $_POST['slgs_screenshot_urls'] ) ) { update_post_meta( $post_id, '_slgs_screenshot_urls', sanitize_textarea_field( $_POST['slgs_screenshot_urls'] ) ); }

    update_post_meta( $post_id, '_slgs_is_singleplayer', isset( $_POST['slgs_is_singleplayer'] ) ? 1 : 0 );
    update_post_meta( $post_id, '_slgs_is_multiplayer', isset( $_POST['slgs_is_multiplayer'] ) ? 1 : 0 );
    update_post_meta( $post_id, '_slgs_is_coop', isset( $_POST['slgs_is_coop'] ) ? 1 : 0 );

    if ( isset( $_POST['slgs_fsk_rating'] ) ) { update_post_meta( $post_id, '_slgs_fsk_rating', sanitize_text_field( $_POST['slgs_fsk_rating'] ) ); }
    if ( isset( $_POST['slgs_supported_languages'] ) ) { update_post_meta( $post_id, '_slgs_supported_languages', sanitize_text_field( $_POST['slgs_supported_languages'] ) ); }

    if ( isset( $_POST['slgs_price'] ) ) { update_post_meta( $post_id, '_slgs_price', sanitize_text_field( $_POST['slgs_price'] ) ); }
    if ( isset( $_POST['slgs_min_os'] ) ) { update_post_meta( $post_id, '_slgs_min_os', sanitize_text_field( $_POST['slgs_min_os'] ) ); }
    if ( isset( $_POST['slgs_min_cpu'] ) ) { update_post_meta( $post_id, '_slgs_min_cpu', sanitize_text_field( $_POST['slgs_min_cpu'] ) ); }
    if ( isset( $_POST['slgs_min_ram'] ) ) { update_post_meta( $post_id, '_slgs_min_ram', sanitize_text_field( $_POST['slgs_min_ram'] ) ); }
    if ( isset( $_POST['slgs_min_gpu'] ) ) { update_post_meta( $post_id, '_slgs_min_gpu', sanitize_text_field( $_POST['slgs_min_gpu'] ) ); }
    if ( isset( $_POST['slgs_rec_os'] ) ) { update_post_meta( $post_id, '_slgs_rec_os', sanitize_text_field( $_POST['slgs_rec_os'] ) ); }
    if ( isset( $_POST['slgs_rec_cpu'] ) ) { update_post_meta( $post_id, '_slgs_rec_cpu', sanitize_text_field( $_POST['slgs_rec_cpu'] ) ); }
    if ( isset( $_POST['slgs_rec_ram'] ) ) { update_post_meta( $post_id, '_slgs_rec_ram', sanitize_text_field( $_POST['slgs_rec_ram'] ) ); }
    if ( isset( $_POST['slgs_rec_gpu'] ) ) { update_post_meta( $post_id, '_slgs_rec_gpu', sanitize_text_field( $_POST['slgs_rec_gpu'] ) ); }
    if ( isset( $_POST['slgs_update_log'] ) ) { update_post_meta( $post_id, '_slgs_update_log', sanitize_textarea_field( $_POST['slgs_update_log'] ) ); }
}
add_action( 'save_post', 'slgs_save_meta_box_data' );

/**
 * Registriert Admin-Skripte und Styles für die Medienbibliothek.
 */
function slgs_enqueue_admin_scripts($hook) {
    // Nur auf den Seiten 'Post bearbeiten' und 'Neuer Post' für den 'game' CPT laden
    if ( ('post.php' != $hook && 'post-new.php' != $hook) || (isset($_GET['post_type']) && 'game' !== $_GET['post_type']) ) {
        return;
    }

    global $typenow;
    if ( $typenow === 'game' ) {
        // Enqueue core media library scripts
        wp_enqueue_media();

        // Registrieren und Enqueue eines eindeutigen Handles, abhängig von jQuery und media-upload (KRITISCHER FEHLER FIX)
        $handle = 'slgs-admin-media-logic';
        wp_register_script( $handle, '', array( 'jquery', 'media-upload' ), '3.1.23', true );
        wp_enqueue_script( $handle );

        // Inline Script für die Medienbibliothek
        $admin_script = "
            jQuery(document).ready(function(\$) {
                var mediaUploader;

                \$('#slgs-upload-screenshots').on('click', function(e) {
                    e.preventDefault();

                    // Wenn der Uploader bereits existiert, öffne ihn erneut.
                    if (mediaUploader) {
                        mediaUploader.open();
                        return;
                    }

                    // Erstelle den neuen Uploader
                    mediaUploader = wp.media({
                        title: 'Screenshots auswählen',
                        button: { text: 'Screenshots verwenden' },
                        multiple: true // Mehrere Dateien erlauben
                    });

                    // Wenn Dateien ausgewählt wurden
                    mediaUploader.on('select', function() {
                        var attachments = mediaUploader.state().get('selection').toJSON();
                        var urls = [];
                        var previewHtml = '';

                        attachments.forEach(function(attachment) {
                            // Verwenden der vollen URL des Mediums
                            var url = attachment.url;
                            urls.push(url);
                            previewHtml += '<img src=\"' + url + '\" style=\"width: 80px; height: 80px; object-fit: cover; border-radius: 4px;\" />';
                        });

                        // URLs mit Zeilenumbrüchen in das Hidden Field speichern
                        \$('#slgs_screenshot_urls').val(urls.join('\\n'));

                        // Vorschau aktualisieren
                        \$('#slgs-screenshots-preview').html(previewHtml);
                        \$('#slgs-clear-screenshots').show();
                    });

                    // Öffne den Uploader
                    mediaUploader.open();
                });

                // Clear-Button Funktionalität
                \$('#slgs-clear-screenshots').on('click', function() {
                    \$('#slgs_screenshot_urls').val('');
                    \$('#slgs-screenshots-preview').empty();
                    \$(this).hide();
                });
            });
        ";
        wp_add_inline_script( 'slgs-admin-media-logic', $admin_script );
    }
}
add_action( 'admin_enqueue_scripts', 'slgs_enqueue_admin_scripts' );


// ===============================================
// 3. AJAX ENDPUNKTE (Bibliothek & RATING & CART)
// ===============================================

function slgs_toggle_cart() {
    if ( ! is_user_logged_in() ) {
        wp_send_json_error( array( 'message' => __( 'Sie müssen angemeldet sein, um den Warenkorb zu verwenden.', 'steam-like-shop' ) ) );
    }

    if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'slgs_cart_nonce' ) ) {
        wp_send_json_error( array( 'message' => 'Sicherheitsprüfung fehlgeschlagen.' ) );
    }

    $post_id = intval( $_POST['post_id'] );
    $user_id = get_current_user_id();
    $cart = get_user_meta( $user_id, '_slgs_user_cart', true );
    if ( ! is_array( $cart ) ) { $cart = array(); }

    $mode = isset($_POST['mode']) ? sanitize_text_field($_POST['mode']) : '';
    $is_in_cart = in_array( $post_id, $cart );
    $action = '';

    if ( $mode === 'remove' || $is_in_cart ) {
        $cart = array_diff( $cart, array( $post_id ) );
        $action = 'removed';
    } else {
        $collection = get_user_meta( $user_id, '_slgs_user_collection', true );
        if ( is_array( $collection ) && in_array( $post_id, $collection ) ) {
             wp_send_json_error( array( 'message' => __( 'Dieses Spiel ist bereits in Ihrer Bibliothek.', 'steam-like-shop' ) ) );
        }

        $cart[] = $post_id;
        $action = 'added';
    }

    $cart = array_values( array_unique( $cart ) );
    update_user_meta( $user_id, '_slgs_user_cart', $cart );

    wp_send_json_success( array( 'action' => $action, 'post_id' => $post_id, 'cart_count' => count($cart) ) );
}
add_action( 'wp_ajax_slgs_toggle_cart', 'slgs_toggle_cart' );

function slgs_process_checkout() {
    if ( ! is_user_logged_in() ) {
        wp_send_json_error( array( 'message' => __( 'Sie müssen angemeldet sein, um den Kauf abzuschließen.', 'steam-like-shop' ) ) );
    }

    if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'slgs_checkout_nonce' ) ) {
        wp_send_json_error( array( 'message' => 'Sicherheitsprüfung fehlgeschlagen.' ) );
    }

    $user_id = get_current_user_id();
    $cart = get_user_meta( $user_id, '_slgs_user_cart', true );
    $collection = get_user_meta( $user_id, '_slgs_user_collection', true );
    $wishlist = get_user_meta( $user_id, '_slgs_user_wishlist', true );

    $cart = is_array( $cart ) ? $cart : array();
    $collection = is_array( $collection ) ? $collection : array();
    $wishlist = is_array( $wishlist ) ? $wishlist : array();

    if ( empty( $cart ) ) {
        wp_send_json_error( array( 'message' => __( 'Ihr Warenkorb ist leer.', 'steam-like-shop' ) ) );
    }

    $purchased_count = 0;
    $total_cost = 0.0;
    $purchased_ids = array();
    $currency_symbol = get_option( 'slgs_currency_symbol', '€' );

    foreach ( $cart as $game_id ) {
        $price = (float) get_post_meta( $game_id, '_slgs_price', true );
        $total_cost += $price;

        if ( $price > 0.00 ) {
            wp_send_json_error( array(
                'message' => __( 'Checkout blockiert: Echte Zahlung für kostenpflichtige Spiele erforderlich. (Simulationsmodus)', 'steam-like-shop' ),
            ) );
        }

        if ( ! in_array( $game_id, $collection ) ) {
            $collection[] = $game_id;
            $purchased_ids[] = $game_id;
            $purchased_count++;
        }
    }

    update_user_meta( $user_id, '_slgs_user_collection', array_values( $collection ) );
    update_user_meta( $user_id, '_slgs_user_cart', array() );
    $wishlist = array_diff( $wishlist, $purchased_ids );
    update_user_meta( $user_id, '_slgs_user_wishlist', array_values( $wishlist ) );

    wp_send_json_success( array(
        'message' => sprintf( __( 'Kauf abgeschlossen! %d Spiele wurden Ihrer Bibliothek hinzugefügt. Gesamtkosten: %s', 'steam-like-shop' ), $purchased_count, number_format($total_cost, 2, ',', '.') . ' ' . esc_html($currency_symbol) ),
        'purchased_ids' => $purchased_ids
    ) );
}
add_action( 'wp_ajax_slgs_process_checkout', 'slgs_process_checkout' );


function slgs_toggle_collection() {
    if ( ! is_user_logged_in() ) { wp_send_json_error( array( 'message' => __( 'Sie müssen angemeldet sein.', 'steam-like-shop' ) ) ); }
    if ( ! isset( $_POST['post_id'] ) || ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'slgs_collection_nonce' ) ) { wp_send_json_error( array( 'message' => 'Sicherheitsprüfung fehlgeschlagen.' ) ); }

    $post_id = intval( $_POST['post_id'] );
    $user_id = get_current_user_id();
    $collection = get_user_meta( $user_id, '_slgs_user_collection', true );
    if ( ! is_array( $collection ) ) { $collection = array(); }

    $mode = isset($_POST['mode']) ? sanitize_text_field($_POST['mode']) : '';
    $is_in_collection = in_array( $post_id, $collection );
    $action = '';

    if ( $mode === 'remove' || $is_in_collection ) {
        $collection = array_diff( $collection, array( $post_id ) );
        $action = 'removed';
    } else {
        wp_send_json_error( array( 'message' => __( 'Fehler: Spiele können nur über den Checkout-Prozess zur Bibliothek hinzugefügt werden.', 'steam-like-shop' ) ) );
    }

    update_user_meta( $user_id, '_slgs_user_collection', array_values( $collection ) );
    wp_send_json_success( array( 'action' => $action, 'post_id' => $post_id ) );
}
add_action( 'wp_ajax_slgs_toggle_collection', 'slgs_toggle_collection' );


function slgs_toggle_wishlist() {
    if ( ! is_user_logged_in() ) { wp_send_json_error( array( 'message' => __( 'Sie müssen angemeldet sein.', 'steam-like-shop' ) ) ); }
    if ( ! isset( $_POST['post_id'] ) || ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'slgs_wishlist_nonce' ) ) { wp_send_json_error( array( 'message' => 'Sicherheitsprüfung fehlgeschlagen.' ) ); }

    $post_id = intval( $_POST['post_id'] );
    $user_id = get_current_user_id();
    $wishlist = get_user_meta( $user_id, '_slgs_user_wishlist', true );
    if ( ! is_array( $wishlist ) ) { $wishlist = array(); }

    $mode = isset($_POST['mode']) ? sanitize_text_field($_POST['mode']) : '';
    $is_on_list = in_array( $post_id, $wishlist );

    if ( $mode === 'remove' || $is_on_list ) {
        $wishlist = array_diff( $wishlist, array( $post_id ) );
        $action = 'removed';
    } else {
        $collection = get_user_meta( $user_id, '_slgs_user_collection', true );
        if ( is_array( $collection ) && in_array( $post_id, $collection ) ) { wp_send_json_error( array( 'message' => __( 'Dieses Spiel ist bereits in Ihrer Bibliothek.', 'steam-like-shop' ) ) ); }

        $wishlist[] = $post_id;
        $action = 'added';
    }

    update_user_meta( $user_id, '_slgs_user_wishlist', array_values( $wishlist ) );
    wp_send_json_success( array( 'action' => $action, 'post_id' => $post_id ) );
}
add_action( 'wp_ajax_slgs_toggle_wishlist', 'slgs_toggle_wishlist' );


function slgs_submit_rating() {
    if ( ! is_user_logged_in() ) { wp_send_json_error( array( 'message' => __( 'Sie müssen angemeldet sein.', 'steam-like-shop' ) ) ); }
    if ( ! isset( $_POST['post_id'] ) || ! isset( $_POST['rating'] ) || ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'slgs_rating_nonce' ) ) { wp_send_json_error( array( 'message' => 'Sicherheitsprüfung fehlgeschlagen.' ) ); }

    $post_id = intval( $_POST['post_id'] );
    $user_id = get_current_user_id();
    $rating = intval( $_POST['rating'] );

    if ( $rating < 1 || $rating > 5 ) { wp_send_json_error( array( 'message' => 'Ungültige Bewertung.' ) ); }

    $collection = get_user_meta( $user_id, '_slgs_user_collection', true );
    if ( ! is_array( $collection ) || ! in_array( $post_id, $collection ) ) { wp_send_json_error( array( 'message' => __( 'Sie müssen das Spiel besitzen, um es bewerten zu können.', 'steam-like-shop' ) ) ) ; }

    $old_rating = (int) get_user_meta( $user_id, '_slgs_rated_game_' . $post_id, true );
    $is_update = $old_rating > 0;

    $rating_sum    = (int) get_post_meta( $post_id, '_slgs_rating_sum', true );
    $total_ratings = (int) get_post_meta( $post_id, '_slgs_total_ratings', true );

    if ($is_update) {
        $new_rating_sum = $rating_sum - $old_rating + $rating;
        $new_total_ratings = $total_ratings;
        $success_message = __( 'Ihre Bewertung wurde erfolgreich aktualisiert!', 'steam-like-shop' );
    } else {
        $new_rating_sum = $rating_sum + $rating;
        $new_total_ratings = $total_ratings + 1;
        $success_message = __( 'Ihre Bewertung wurde erfolgreich gespeichert!', 'steam-like-shop' );
    }

    $new_average_rating = ($new_total_ratings > 0) ? round( $new_rating_sum / $new_total_ratings, 1 ) : 0;

    update_post_meta( $post_id, '_slgs_rating_sum', $new_rating_sum );
    update_post_meta( $post_id, '_slgs_total_ratings', $new_total_ratings );
    update_post_meta( $post_id, '_slgs_average_rating', $new_average_rating );

    update_user_meta( $user_id, '_slgs_rated_game_' . $post_id, $rating );

    wp_send_json_success( array(
        'message' => $success_message,
        'average' => number_format($new_average_rating, 1, ',', ''),
        'total'   => number_format_i18n($new_total_ratings),
        'rating'  => $rating
    ) );
}
function slgs_track_activation() {
    if ( ! isset( $_POST['post_id'] ) || ! isset( $_POST['nonce'] ) ) {
        wp_send_json_error( array( 'message' => 'Fehlende Parameter.' ) );
    }

    if ( ! wp_verify_nonce( $_POST['nonce'], 'slgs_activation_nonce' ) ) {
        wp_send_json_error( array( 'message' => 'Sicherheitsprfung fehlgeschlagen.' ) );
    }

    $post_id = intval( $_POST['post_id'] );

    if ( $post_id <= 0 ) {
        wp_send_json_error( array( 'message' => 'Ungltige Spiel-ID.' ) );
    }

    $current_count = (int) get_post_meta( $post_id, '_slgs_activation_count', true );
    $new_count = $current_count + 1;

    update_post_meta( $post_id, '_slgs_activation_count', $new_count );

    wp_send_json_success( array(
        'message' => 'Aktivierung gezhlt',
        'new_count' => $new_count
    ) );
}
add_action( 'wp_ajax_slgs_track_activation', 'slgs_track_activation' );
add_action( 'wp_ajax_nopriv_slgs_track_activation', 'slgs_track_activation' );
add_action( 'wp_ajax_slgs_submit_rating', 'slgs_submit_rating' );


// ===============================================
// 4. SHOP & FEATURED SHORTCODES
// ===============================================

function slgs_get_user_cart_count() {
    if (!is_user_logged_in()) { return 0; }
    $cart = get_user_meta(get_current_user_id(), '_slgs_user_cart', true);
    return is_array($cart) ? count($cart) : 0;
}

function slgs_get_cart_widget() {
    $count = slgs_get_user_cart_count();

    $is_active = $count > 0 ? 'slgs-cart-active' : '';
    $slug = get_option('slgs_checkout_page_slug', 'checkout');
    $checkout_url = home_url( '/' . $slug );

    if (!is_user_logged_in()) { $checkout_url = wp_login_url( $checkout_url ); }

    $html = '<a href="' . esc_url($checkout_url) . '" class="slgs-cart-widget ' . esc_attr($is_active) . '" title="' . esc_attr__('Zum Warenkorb', 'steam-like-shop') . '" id="slgs-cart-widget">';
    $html .= '<span class="dashicons dashicons-cart"></span>';
    $html .= '<span class="slgs-cart-count">' . esc_html($count) . '</span>';
    $html .= '</a>';

    return $html;
}

function slgs_get_feature_icons( $post_id ) {
    $html = '';
    $is_multiplayer  = get_post_meta( $post_id, '_slgs_is_multiplayer', true );
    $is_singleplayer = get_post_meta( $post_id, '_slgs_is_singleplayer', true );
    $is_coop         = get_post_meta( $post_id, '_slgs_is_coop', true );
    $fsk = get_post_meta( $post_id, '_slgs_fsk_rating', true );

    if ($is_multiplayer) { $html .= '<span class="slgs-feature-icon" title="' . esc_attr__( 'Mehrspieler', 'steam-like-shop' ) . '"><span class="dashicons dashicons-groups"></span></span>'; }
    if ($is_singleplayer) { $html .= '<span class="slgs-feature-icon" title="' . esc_attr__( 'Einzelspieler', 'steam-like-shop' ) . '"><span class="dashicons dashicons-admin-users"></span></span>'; }
    if ($is_coop) { $html .= '<span class="slgs-feature-icon" title="' . esc_attr__( 'Kooperativ', 'steam-like-shop' ) . '"><span class="dashicons dashicons-share-alt2"></span></span>'; }

    if ($fsk) {
        $fsk_number = (int) filter_var($fsk, FILTER_SANITIZE_NUMBER_INT);
        $html .= '<span class="slgs-fsk-badge slgs-fsk-' . esc_attr($fsk_number) . '">' . esc_html($fsk_number) . '</span>';
    }

    return '<div class="slgs-game-features">' . $html . '</div>';
}

/**
 * Generiert den Konsistenten Header-Bereich mit Titel, Warenkorb und Navigation.
 * Dies ersetzt die statischen Navigationsleisten auf Einzel- und Warenkorbseiten.
 * * @param string $title_override Optionaler Titel, falls nicht der Standard-Shop-Titel verwendet werden soll.
 * @param string $context 'hub', 'single', oder 'cart'.
 * @return string Der generierte HTML-Code.
 */
function slgs_render_consistent_header( $title_override = '', $context = 'hub' ) {
    $shop_title = get_option( 'slgs_shop_title', __( 'Game Shop Übersicht', 'steam-like-shop' ) );
    $title = $title_override ?: $shop_title;

    ob_start();
    ?>
    <div class="slgs-consistent-header slgs-header-<?php echo esc_attr($context); ?>">
        <div class="slgs-main-hub-header">
            <h2><?php echo esc_html( $title ); ?></h2>
            <?php echo slgs_get_cart_widget(); ?>
        </div>
        <?php echo slgs_render_nav_bar( $context ); ?>
    </div>
    <?php
    return ob_get_clean();
}

/**
 * Generiert die Hauptnavigation des Shops. (Erweitert um Dropdown für "Store")
 *
 * @param string $context 'hub', 'single', oder 'cart'.
 * @return string Der generierte HTML-Code für die Navigation.
 */
function slgs_render_nav_bar( $context = 'hub' ) {
    $tabs = array(
        'shop' => __( 'Store', 'steam-like-shop' ), 'new' => __( 'Neu', 'steam-like-shop' ),
        'top' => __( 'Meistgespielt', 'steam-like-shop' ), 'best_rated' => __( 'Top-Bewertet', 'steam-like-shop' ),
    );
    $is_user_logged_in = is_user_logged_in();
    if ( $is_user_logged_in ) {
        $tabs['my_games'] = __( 'Bibliothek', 'steam-like-shop' );
        $tabs['wishlist'] = __( 'Wunschliste', 'steam-like-shop' );
    }

    // Haupt-Shop-URL aus den Einstellungen abrufen
    $base_url = get_option( 'slgs_main_shop_url', home_url( '/' ) );
    $categories = get_terms( array( 'taxonomy' => 'game_category', 'hide_empty' => true, ) );

    ob_start();
    echo '<nav class="slgs-tab-nav slgs-nav-' . esc_attr($context) . '">';

    foreach ( $tabs as $key => $label ) {
        // Im Hub ist der 'shop' Tab standardmäßig aktiv, falls keine URL-Hash gesetzt ist
        $active_class = ( $context === 'hub' && $key === 'shop' ) ? ' slgs-tab-active' : '';

        if ($key === 'shop' && $context === 'hub' && !empty($categories)) { // Dropdown nur im Hub und wenn Kategorien existieren
            echo '<div class="slgs-dropdown-menu' . esc_attr( $active_class ) . '">';
            // Store-Button, der das Dropdown öffnet, mit Tab-Funktion für 'all' games
            echo '<button class="slgs-tab-button slgs-dropdown-toggle' . esc_attr( $active_class ) . '" data-tab="shop" data-term="all">' . esc_html( $label ) . ' <span class="dashicons dashicons-arrow-down"></span></button>';

            echo '<div class="slgs-dropdown-content">';
            // Option 1: Alle Spiele (Default des Shop Tabs)
            echo '<button class="slgs-tab-button slgs-category-link" data-tab="shop" data-term="all">' . __( 'Alle Spiele', 'steam-like-shop' ) . '</button>';
            // Optionen 2: Kategorien (nur für den Shop-Tab)
            foreach ( $categories as $category ) {
                 echo '<button class="slgs-tab-button slgs-category-link" data-tab="shop" data-term="' . esc_attr( $category->slug ) . '">' . esc_html( $category->name ) . '</button>';
            }
            echo '</div>';
            echo '</div>';
        } else {
            // Normale Buttons/Links für andere Tabs
            if ($context === 'hub') {
                echo '<button class="slgs-tab-button' . esc_attr( $active_class ) . '" data-tab="' . esc_attr( $key ) . '">' . esc_html( $label ) . '</button>';
            } else {
                // Auf der Einzelansicht/Cart: Verwenden Sie Links zur Hauptseite mit Hash-Fragment
                $url = esc_url( $base_url . '#slgs-tab-' . $key );
                echo '<a href="' . $url . '" class="slgs-tab-button">' . esc_html( $label ) . '</a>';
            }
        }
    }

    echo '</nav>';
    return ob_get_clean();
}

/**
 * Gibt eine CSS-Klasse basierend auf der Durchschnittsbewertung (0-5) zurück.
 * @param float $rating
 * @return string
 */
function slgs_get_rating_color_class( $rating ) {
    $rating = max(0, min(5, (float)$rating));
    if ($rating >= 4.0) return 'slgs-rating-excellent';
    if ($rating >= 3.0) return 'slgs-rating-good';
    if ($rating >= 2.0) return 'slgs-rating-mixed';
    if ($rating >= 1.0) return 'slgs-rating-bad';
    return 'slgs-rating-none';
}


function slgs_display_games( $list_type = 'shop', $show_title = true ) {
    ob_start();

    $args = array(
        'post_type'      => 'game',
        'post_status'    => 'publish',
        'posts_per_page' => ( $list_type === 'shop' ? -1 : 6 ),
    );

    $title = '';
    $message = '';

    // Die Standard-Post-Filterung für den Shop-Tab (nur nicht im Besitz befindliche Spiele)
    if ( $list_type === 'shop' && is_user_logged_in() ) {
        $user_id = get_current_user_id();
        $collection = get_user_meta( $user_id, '_slgs_user_collection', true );
        $owned_ids = is_array( $collection ) ? $collection : array();
        if ( ! empty( $owned_ids ) ) { $args['post__not_in'] = $owned_ids; }
    }


    switch ( $list_type ) {
        case 'shop':
            $title = __( 'Alle Spiele im Shop', 'steam-like-shop' );
            break;
        case 'new':
            $args['orderby'] = 'date'; $args['order'] = 'DESC';
            break;
        case 'top':
            $args['orderby'] = 'meta_value_num'; $args['meta_key'] = '_slgs_activation_count'; $args['order'] = 'DESC';
            $args['meta_query'] = array( array( 'key' => '_slgs_activation_count', 'value' => 0, 'compare' => '>', 'type' => 'NUMERIC' ) );
            break;
        case 'best_rated':
            $args['orderby'] = 'meta_value_num'; $args['meta_key'] = '_slgs_average_rating'; $args['order'] = 'DESC';
            $args['meta_query'] = array( array( 'key' => '_slgs_total_ratings', 'value' => 0, 'compare' => '>', 'type' => 'NUMERIC' ) );
            break;

        case 'my_games':
        case 'wishlist':
        case 'my_wishlist':
        case 'cart':

            if ( ! is_user_logged_in() ) {
                if ($show_title) { echo '<p class="slgs-warning">' . __( 'Sie müssen angemeldet sein, um diese Liste anzuzeigen.', 'steam-like-shop' ) . '</p>'; }
                return ob_get_clean();
            }

            $user_id = get_current_user_id();

            // Korrekte Zuweisung basierend auf dem Listentyp
            if ($list_type === 'cart') {
                $meta_key = '_slgs_user_cart';
                $message = __( 'Ihr Warenkorb ist leer.', 'steam-like-shop' );
                $title = __( 'Ihr Warenkorb', 'steam-like-shop' );
            } elseif ($list_type === 'wishlist' || $list_type === 'my_wishlist') {
                $meta_key = '_slgs_user_wishlist';
                $message = __( 'Ihre Wunschliste ist leer.', 'steam-like-shop' );
            } else { // 'my_games'
                $meta_key = '_slgs_user_collection';
                $message = __( 'Ihre Bibliothek ist leer. Kaufen oder beanspruchen Sie Spiele im Store!', 'steam-like-shop' );

            }

            $list_data = get_user_meta( $user_id, $meta_key, true );
            $game_ids = is_array( $list_data ) ? $list_data : array();

            if ( empty( $game_ids ) ) {
                echo '<p class="slgs-info">' . $message . '</p>';
                return ob_get_clean();
            }


            $args['post__in'] = $game_ids;
            $args['orderby'] = 'post__in';
            break;
    }

    if ( $show_title ) { echo '<h2>' . esc_html( $title ) . '</h2>'; }

    // Filter-Container wird nicht mehr direkt in der Shop-Liste angezeigt
    if ( $list_type === 'shop' ) { echo '<div class="slgs-filter-container" style="display:none;"></div>'; }


    $games_query = new WP_Query( $args );

    echo '<div class="slgs-game-list slgs-list-' . esc_attr( $list_type ) . '">';

    if ( $games_query->have_posts() ) {
        $currency_symbol = get_option( 'slgs_currency_symbol', '€' );
        while ( $games_query->have_posts() ) {
            $games_query->the_post();
            $post_id = get_the_ID();
            $creator = get_post_meta( $post_id, '_slgs_creator', true );
            $game_terms = get_the_terms( $post_id, 'game_category' );
            $term_classes = $game_terms ? implode( ' ', wp_list_pluck( $game_terms, 'slug' ) ) : '';

            $rating_display = slgs_get_rating_stars( get_post_meta( $post_id, '_slgs_average_rating', true ) );
            $feature_icons = slgs_get_feature_icons($post_id);
            $price = get_post_meta( $post_id, '_slgs_price', true );
            $price_display = empty($price) || floatval($price) == 0.00 ? __( 'Kostenlos', 'steam-like-shop' ) : number_format_i18n( floatval($price), 2 ) . ' ' . esc_html($currency_symbol);

            $remove_title = ''; $remove_action = ''; $remove_nonce_key = '';
            if ($list_type === 'my_games') { $remove_title = __( 'Aus Bibliothek entfernen', 'steam-like-shop' ); $remove_action = 'slgs_toggle_collection'; $remove_nonce_key = 'collection_nonce'; }
            elseif ($list_type === 'wishlist' || $list_type === 'my_wishlist') { $remove_title = __( 'Von Wunschliste entfernen', 'steam-like-shop' ); $remove_action = 'slgs_toggle_wishlist'; $remove_nonce_key = 'wishlist_nonce'; }
            elseif ($list_type === 'cart') { $remove_title = __( 'Aus Warenkorb entfernen', 'steam-like-shop' ); $remove_action = 'slgs_toggle_cart'; $remove_nonce_key = 'cart_nonce'; }
            ?>
            <div class="slgs-game-card <?php echo esc_attr( $term_classes ); ?>" data-post-id="<?php echo esc_attr( $post_id ); ?>">
                <?php if ($list_type === 'my_games' || $list_type === 'wishlist' || $list_type === 'my_wishlist'): ?> <button class="slgs-card-remove-icon" title="<?php echo esc_attr($remove_title); ?>" data-post-id="<?php echo esc_attr($post_id); ?>" data-action="<?php echo esc_attr($remove_action); ?>" data-nonce-key="<?php echo esc_attr($remove_nonce_key); ?>" data-list-type="<?php echo esc_attr($list_type); ?>">
                            <span class="dashicons dashicons-trash"></span>
                        </button>
                    <?php endif; ?>

                <?php if ( has_post_thumbnail() ) : ?><a href="<?php the_permalink(); ?>" class="slgs-card-image-link"><?php the_post_thumbnail( 'medium', array( 'class' => 'slgs-card-image' ) ); ?></a><?php endif; ?>
                <div class="slgs-card-content">
                    <h3 class="slgs-card-title"><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h3>
                    <div class="slgs-card-rating"><?php echo $rating_display; ?></div>
                    <p class="slgs-card-creator"><?php echo __( 'Ersteller: ', 'steam-like-shop' ) . esc_html( $creator ); ?></p>
                    <?php echo $feature_icons; ?>
                    <p class="slgs-card-excerpt"><?php echo get_the_excerpt(); ?></p>
                    <div class="slgs-card-price"><?php echo esc_html( $price_display ); ?></div>
                    <a href="<?php the_permalink(); ?>" class="slgs-details-button"><?php _e( 'Details ansehen', 'steam-like-shop' ); ?></a>
                </div>
            </div>
            <?php
        }
        wp_reset_postdata();
    } else {
        echo '<p class="slgs-info">' . ( $list_type === 'shop' ? __( 'Alle Spiele sind bereits in Ihrer Bibliothek oder es sind keine Spiele im Store vorhanden.', 'steam-like-shop' ) : $message ) . '</p>';
    }

    echo '</div>';
    return ob_get_clean();
}

function slgs_main_hub_shortcode() {
    ob_start();
    $shop_title = get_option( 'slgs_shop_title', __( 'Game Shop Übersicht', 'steam-like-shop' ) );

    echo '<div class="slgs-main-hub">';
    // Verwendet den konsistenten Header für den Hub
    echo slgs_render_consistent_header( $shop_title, 'hub' );

    // Die AJAX-Tabs
    $tabs = array(
        'shop', 'new', 'top', 'best_rated', 'my_games', 'wishlist'
    );
    $is_user_logged_in = is_user_logged_in();
    $active_tab = 'shop'; // Standardmäßig 'Store'

    echo '<div class="slgs-tab-content-wrapper">';
    foreach ( $tabs as $key ) {
        if (($key === 'my_games' || $key === 'wishlist') && !$is_user_logged_in) { continue; }
        $display_style = ( $key !== $active_tab ) ? 'style="display: none;"' : '';
        echo '<div id="slgs-tab-' . esc_attr( $key ) . '" class="slgs-tab-panel" data-tab-key="' . esc_attr($key) . '" ' . $display_style . '>';
        // Lädt Spiele/Informationen für alle Tabs
        echo slgs_display_games( $key, false );
        echo '</div>';
    }

    if ( ! $is_user_logged_in ) {
         echo '<div id="slgs-tab-my_games" class="slgs-tab-panel" data-tab-key="my_games" style="display: none;"><p class="slgs-warning slgs-large-warning">' . __( 'Bitte melden Sie sich an, um Spiele zu Ihrer Bibliothek hinzuzufügen und diesen Bereich zu sehen.', 'steam-like-shop' ) . '</p></div>';
         echo '<div id="slgs-tab-wishlist" class="slgs-tab-panel" data-tab-key="wishlist" style="display: none;"><p class="slgs-warning slgs-large-warning">' . __( 'Bitte melden Sie sich an, um Ihre Wunschliste zu sehen und zu verwalten.', 'steam-like-shop' ) . '</p></div>';
    }

    echo '</div>';
    echo '</div>';

    return ob_get_clean();
}
add_shortcode( 'slgs_main_hub', 'slgs_main_hub_shortcode' );

add_shortcode( 'best_rated_games', 'slgs_best_rated_games_shortcode' ); function slgs_best_rated_games_shortcode() { return slgs_display_games( 'best_rated', true ); }
add_shortcode( 'game_shop', 'slgs_game_shop_shortcode' ); function slgs_game_shop_shortcode() { return slgs_display_games( 'shop', true ); }
add_shortcode( 'new_games', 'slgs_new_games_shortcode' ); function slgs_new_games_shortcode() { return slgs_display_games( 'new', true ); }
add_shortcode( 'top_games', 'slgs_top_games_shortcode' ); function slgs_top_games_shortcode() { return slgs_display_games( 'top', true ); }
add_shortcode( 'my_games', 'slgs_my_games_shortcode' ); function slgs_my_games_shortcode() { return slgs_display_games( 'my_games', true ); }
add_shortcode( 'my_wishlist', 'slgs_my_wishlist_shortcode' ); function slgs_my_wishlist_shortcode() { return slgs_display_games( 'my_wishlist', true ); }
add_shortcode( 'slgs_cart_button', 'slgs_cart_button_shortcode' ); function slgs_cart_button_shortcode() { return slgs_get_cart_widget(); }
add_shortcode( 'latest_reviews', 'slgs_latest_reviews_shortcode' ); function slgs_latest_reviews_shortcode() { ob_start(); echo '<h2>' . __( 'Neueste Community-Aktivität', 'steam-like-shop' ) . '</h2>'; echo '<p class="slgs-info">' . __( 'Hinweis: Text-Rezensionen wurden entfernt.', 'steam-like-shop' ) . '</p>'; return ob_get_clean(); }

function slgs_cart_checkout_shortcode() {
    ob_start();

    // Warenkorb-Titel
    $cart_title = __( 'Ihr Warenkorb', 'steam-like-shop' );

    // Konsistenten Header (mit Navigation) anzeigen
    echo '<div class="slgs-cart-checkout-container">';
    echo slgs_render_consistent_header( $cart_title, 'cart' );


    if ( ! is_user_logged_in() ) {
        echo '<p class="slgs-warning slgs-large-warning">' . __( 'Bitte melden Sie sich an, um Ihren Warenkorb anzuzeigen und zur Kasse zu gehen.', 'steam-like-shop' ) . '</p>';
        echo '</div>';
        return ob_get_clean();
    }

    $user_id = get_current_user_id();
    $cart = get_user_meta( $user_id, '_slgs_user_cart', true );
    $game_ids = is_array( $cart ) ? $cart : array();
    $currency_symbol = get_option( 'slgs_currency_symbol', '€' );
    $total_cost = 0.0;

    // URL für den "Einkauf fortsetzen" Button (aus Einstellungen)
    $continue_shopping_url = get_option( 'slgs_continue_shopping_url', get_post_type_archive_link( 'game' ) );


    if ( empty( $game_ids ) ) {
        echo '<p class="slgs-info">' . __( 'Ihr Warenkorb ist leer.', 'steam-like-shop' ) . '</p>';
        // Konsolidierter "Einkauf fortsetzen" Button für leeren Warenkorb
        echo '<div class="slgs-checkout-actions slgs-checkout-empty">';
        echo '<a href="' . esc_url( $continue_shopping_url ) . '" class="slgs-continue-shopping-button"><span class="dashicons dashicons-undo"></span> ' . __( 'Einkauf fortsetzen', 'steam-like-shop' ) . '</a>';
        echo '</div>';
    } else {
        $args = array('post_type' => 'game', 'post_status' => 'publish', 'posts_per_page' => -1, 'post__in' => $game_ids, 'orderby' => 'post__in');
        $games_query = new WP_Query( $args );

        echo '<div class="slgs-cart-items-list" id="slgs-cart-items-list">';

        if ( $games_query->have_posts() ) {
             while ( $games_query->have_posts() ) {
                $games_query->the_post();
                $post_id = get_the_ID();
                $price = (float) get_post_meta( $post_id, '_slgs_price', true );
                $total_cost += $price;

                $price_display = empty($price) || $price == 0.00 ? __( 'Kostenlos', 'steam-like-shop' ) : number_format_i18n( $price, 2 ) . ' ' . esc_html($currency_symbol);
                $is_free = $price == 0.00;

                ?>
                <div class="slgs-cart-item" data-post-id="<?php echo esc_attr( $post_id ); ?>" data-price="<?php echo esc_attr( $price ); ?>">
                    <div class="slgs-cart-item-image"><?php if ( has_post_thumbnail() ) : ?><?php the_post_thumbnail( array( 60, 60 ), array( 'class' => 'slgs-cart-thumb' ) ); ?><?php endif; ?></div>
                    <div class="slgs-cart-item-info"><a href="<?php the_permalink(); ?>" class="slgs-cart-item-title"><?php the_title(); ?></a><span class="slgs-cart-item-price <?php echo $is_free ? 'slgs-free' : 'slgs-paid'; ?>"><?php echo esc_html( $price_display ); ?></span></div>
                    <button class="slgs-cart-item-remove" title="<?php _e( 'Aus Warenkorb entfernen', 'steam-like-shop' ); ?>" data-post-id="<?php echo esc_attr( $post_id ); ?>" data-nonce-key="cart_nonce" data-action="slgs_toggle_cart"><span class="dashicons dashicons-trash"></span></button>
                </div>
                <?php
            }
            wp_reset_postdata();
        }

        echo '</div>';

        // Checkout Aktionen (inkl. 'Einkauf fortsetzen' links)
        echo '<div class="slgs-checkout-actions">';
        echo '<a href="' . esc_url( $continue_shopping_url ) . '" class="slgs-continue-shopping-button"><span class="dashicons dashicons-undo"></span> ' . __( 'Einkauf fortsetzen', 'steam-like-shop' ) . '</a>';
        echo '</div>';

        $total_cost_display = number_format( $total_cost, 2, ',', '.' ) . ' ' . esc_html($currency_symbol);
        $checkout_text = ( $total_cost == 0.00 ) ? __( 'Kostenlose Spiele beanspruchen (0,00 ', 'steam-like-shop' ) . esc_html($currency_symbol) . ')' : __( 'Kauf abschließen (Zahlung erforderlich)', 'steam-like-shop' );
        $checkout_class = ( $total_cost == 0.00 ) ? 'slgs-checkout-button slgs-checkout-free' : 'slgs-checkout-button slgs-checkout-paid slgs-disabled';
        $checkout_disabled = ( $total_cost > 0.00 ) ? 'disabled' : '';

        echo '<div class="slgs-cart-summary">';
        echo '<p class="slgs-cart-total"><strong>' . __( 'Gesamtsumme:', 'steam-like-shop' ) . '</strong> <span id="slgs-total-cost" class="slgs-total-cost-value">' . $total_cost_display . '</span></p>';
        echo '<button id="slgs-checkout-button" class="' . esc_attr($checkout_class) . '" ' . esc_attr($checkout_disabled) . ' data-nonce="' . esc_attr(wp_create_nonce( 'slgs_checkout_nonce' )) . '" data-total-raw="' . esc_attr( $total_cost ) . '">';
        echo '<span class="dashicons dashicons-saved"></span> <span id="slgs-checkout-text">' . esc_html($checkout_text) . '</span>';
        echo '</button>';

        if ( $total_cost > 0.00 ) { echo '<p class="slgs-warning-small">' . __( 'HINWEIS: Im Simulationsmodus können nur kostenlose Spiele gekauft werden.', 'steam-like-shop' ) . '</p>'; }

        echo '<p class="slgs-checkout-message" id="slgs-checkout-message"></p>';
        echo '</div>';
    }


    echo '</div>'; // End slgs-cart-checkout-container
    return ob_get_clean();
}
add_shortcode( 'slgs_cart_checkout', 'slgs_cart_checkout_shortcode' );


// ===============================================
// 5. SPIEL-DETAILSEITE (Erweiterte Anpassung mit Galerie)
// ===============================================

function slgs_get_rating_stars( $rating ) {
    $rating = max(0, min(5, (float)$rating));
    $output = '';
    for ($i = 1; $i <= 5; $i++) {
        if ($rating >= $i) { $output .= '<span class="dashicons dashicons-star-filled slgs-star-filled"></span>'; }
        elseif ($rating > ($i - 1) && $rating < $i) {
            if (abs($rating - ($i - 0.5)) < abs($rating - $i)) { $output .= '<span class="dashicons dashicons-star-half slgs-star-half"></span>'; }
            else { $output .= '<span class="dashicons dashicons-star-empty slgs-star-empty"></span>'; }
        } else { $output .= '<span class="dashicons dashicons-star-empty slgs-star-empty"></span>'; }
    }
    return $output;
}

function slgs_get_rating_stars_interactive( $rating, $interactive = true ) {
    $output = '';
    $class = $interactive ? 'slgs-star-interactive' : 'slgs-star-static';
    for ($i = 1; $i <= 5; $i++) {
        $icon = ($rating >= $i) ? 'dashicons-star-filled' : 'dashicons-star-empty';
        $output .= '<span class="dashicons ' . esc_attr($icon) . ' ' . esc_attr($class) . '" data-rating="' . esc_attr($i) . '"></span>';
    }
    return $output;
}


function slgs_display_game_content( $content ) {
    if ( is_singular( 'game' ) && in_the_loop() && is_main_query() ) {
        $post_id = get_the_ID();
        $creator = get_post_meta( $post_id, '_slgs_creator', true );
        $trailer_url = get_post_meta( $post_id, '_slgs_trailer_url', true );
        $screenshot_urls_raw = get_post_meta( $post_id, '_slgs_screenshot_urls', true );
        $game_url = get_post_meta( $post_id, '_slgs_game_url', true );
        $download_url = get_post_meta( $post_id, '_slgs_download_url', true );

        $screenshots = array_filter( array_map( 'trim', explode( "\n", $screenshot_urls_raw ) ) );

        // NEU: Featured Image als ersten Screenshot hinzufügen (falls vorhanden)
        $main_media_url = '';
        if ( has_post_thumbnail() ) {
            $thumbnail_id = get_post_thumbnail_id( $post_id );
            $main_media_url = wp_get_attachment_url( $thumbnail_id );
        }
        if ( !empty($main_media_url) && !in_array($main_media_url, $screenshots) ) {
            array_unshift($screenshots, $main_media_url);
        }
        $initial_image = !empty($screenshots) ? esc_url($screenshots[0]) : '';
        $has_media = !empty($initial_image);


        $activation_count = (int) get_post_meta( $post_id, '_slgs_activation_count', true );
        $average_rating = get_post_meta( $post_id, '_slgs_average_rating', true );
        $rating_display = slgs_get_rating_stars( $average_rating );
        $total_ratings = (int) get_post_meta( $post_id, '_slgs_total_ratings', true );
        $rating_color_class = slgs_get_rating_color_class( $average_rating ); // NEU: Farbbewertung

        $is_multiplayer  = get_post_meta( $post_id, '_slgs_is_multiplayer', true );
        $is_singleplayer = get_post_meta( $post_id, '_slgs_is_singleplayer', true );
        $is_coop         = get_post_meta( $post_id, '_slgs_is_coop', true );
        $fsk_rating      = get_post_meta( $post_id, '_slgs_fsk_rating', true );
        $languages       = array_filter( array_map('trim', explode(',', get_post_meta( $post_id, '_slgs_supported_languages', true ))));
        $update_log_raw  = get_post_meta( $post_id, '_slgs_update_log', true );

        $price           = get_post_meta( $post_id, '_slgs_price', true );
        $currency_symbol = get_option( 'slgs_currency_symbol', '€' );
        $min_os = get_post_meta( $post_id, '_slgs_min_os', true ); $min_cpu = get_post_meta( $post_id, '_slgs_min_cpu', true ); $min_ram = get_post_meta( $post_id, '_slgs_min_ram', true ); $min_gpu = get_post_meta( $post_id, '_slgs_min_gpu', true );
        $rec_os = get_post_meta( $post_id, '_slgs_rec_os', true ); $rec_cpu = get_post_meta( $post_id, '_slgs_rec_cpu', true ); $rec_ram = get_post_meta( $post_id, '_slgs_rec_ram', true ); $rec_gpu = get_post_meta( $post_id, '_slgs_rec_gpu', true );
        $has_recommended = !empty($rec_os) || !empty($rec_cpu) || !empty($rec_ram) || !empty($rec_gpu);
        $has_minimum = !empty($min_os) || !empty($min_cpu) || !empty($min_ram) || !empty($min_gpu);

        $price_float = floatval($price);
        $price_display = empty($price) || $price_float == 0.00 ? __( 'Kostenlos', 'steam-like-shop' ) : number_format($price_float, 2, ',', '.') . ' ' . esc_html($currency_symbol);

        $is_user_logged_in = is_user_logged_in();
        $user_has_rated = false; $is_in_collection = false; $is_on_wishlist = false; $is_in_cart = false;
        $user_rating = 0;

        if ( $is_user_logged_in ) {
            $user_id = get_current_user_id();
            $collection = get_user_meta( $user_id, '_slgs_user_collection', true ); $collection = is_array( $collection ) ? $collection : array(); $is_in_collection = in_array( $post_id, $collection );
            $wishlist = get_user_meta( $user_id, '_slgs_user_wishlist', true ); $wishlist = is_array( $wishlist ) ? $wishlist : array(); $is_on_wishlist = in_array( $post_id, $wishlist );
            $cart = get_user_meta( $user_id, '_slgs_user_cart', true ); $cart = is_array( $cart ) ? $cart : array(); $is_in_cart = in_array( $post_id, $cart );
            $user_rating = (int) get_user_meta( $user_id, '_slgs_rated_game_' . $post_id, true ); $user_has_rated = $user_rating > 0;
        }

        $checkout_url = home_url( '/' . get_option('slgs_checkout_page_slug', 'checkout') );

        ob_start();

        echo '<div class="slgs-single-details">';
        // Konsistenten Header (mit Navigation) anzeigen
        echo slgs_render_consistent_header( get_the_title(), 'single' );

        echo '<div class="slgs-top-section">';

        // NEU: Media-Container mit Galerie
        echo '<div class="slgs-media-container">';
        if ( $has_media ) {
            echo '<div class="slgs-main-media-viewer">';
            echo '<img src="' . $initial_image . '" alt="' . esc_attr(get_the_title()) . ' Media" class="slgs-main-screenshot" id="slgs-main-screenshot">';
            echo '</div>'; // Ende slgs-main-media-viewer

            // 2. Thumbnail-Galerie darunter
            echo '<div class="slgs-media-gallery-thumbs" id="slgs-media-gallery-thumbs">';
            foreach ( $screenshots as $index => $url ) {
                $active_class = ($index === 0) ? ' slgs-thumb-active' : '';
                echo '<img src="' . esc_url( $url ) . '" alt="Thumbnail ' . ($index + 1) . '" class="slgs-media-thumb' . $active_class . '" data-full-url="' . esc_url( $url ) . '">';
            }
            echo '</div>'; // Ende slgs-media-gallery-thumbs
        } else {
            echo '<div class="slgs-placeholder-media">' . __( 'Keine Medien verfügbar', 'steam-like-shop' ) . '</div>';
        }
        echo '</div>'; // Ende slgs-media-container

        echo '<div class="slgs-side-panel">';

        echo '<div class="slgs-metadata-box">';
        echo '<p><strong>' . __( 'Ersteller:', 'steam-like-shop' ) . '</strong> ' . esc_html( $creator ) . '</p>';
        echo '<p><strong>' . __( 'Veröffentlicht am:', 'steam-like-shop' ) . '</strong> ' . esc_html( get_the_date( 'd.m.Y' ) ) . '</p>';
        echo '<p><strong>' . __( 'Kategorien:', 'steam-like-shop' ) . '</strong> ' . get_the_term_list( $post_id, 'game_category', '', ', ' ) . '</p>';
        if ($fsk_rating) { $fsk_number = (int) filter_var($fsk_rating, FILTER_SANITIZE_NUMBER_INT); echo '<p><strong>' . __( 'Altersfreigabe:', 'steam-like-shop' ) . '</strong> <span class="slgs-fsk-tag-large slgs-fsk-' . esc_attr($fsk_number) . '">' . esc_html($fsk_rating) . '</span></p>'; }
        echo '</div>';


        echo '<div class="slgs-metadata-box slgs-feature-list">';
        echo '<p><strong>' . __( 'Gameplay Features:', 'steam-like-shop' ) . '</strong></p>';
        $features_or_languages_found = false;
        if ($is_singleplayer) { echo '<span class="slgs-feature-item"><span class="dashicons dashicons-admin-users"></span> ' . __( 'Einzelspieler', 'steam-like-shop' ) . '</span>'; $features_or_languages_found = true; }
        if ($is_multiplayer) { echo '<span class="slgs-feature-item"><span class="dashicons dashicons-groups"></span> ' . __( 'Mehrspieler', 'steam-like-shop' ) . '</span>'; $features_or_languages_found = true; }
        if ($is_coop) { echo '<span class="slgs-feature-item"><span class="dashicons dashicons-share-alt2"></span> ' . __( 'Kooperativ', 'steam-like-shop' ) . '</span>'; $features_or_languages_found = true; }
        if (!empty($languages)) {
            echo '<div class="slgs-language-feature-block"><p><strong>' . __( 'Sprachen:', 'steam-like-shop' ) . '</strong></p><div class="slgs-languages-list">';
            foreach ($languages as $lang) { echo '<span class="slgs-language-tag">' . esc_html($lang) . '</span>'; }
            echo '</div></div>'; $features_or_languages_found = true;
        }
        if (!$features_or_languages_found) { echo '<p class="slgs-warning-small">' . __( 'Keine spezifischen Features oder Sprachen angegeben.', 'steam-like-shop' ) . '</p>'; }
        echo '</div>';

        // BITTE ERSETZEN SIE DEN OBEN GENANNTEN BLOCK MIT DIESEM NEUEN CODE:
// Dieser neue Block stellt sicher, dass der Kauf-Button immer da ist, wenn das Spiel NICHT in der Sammlung ist.

echo '<div class="slgs-play-section">';
echo '<p class="slgs-game-price-large">' . __( 'Preis:', 'steam-like-shop' ) . ' <strong class="slgs-price-value">' . $price_display . '</strong></p>';

// --- 1. LOGIK: IST DAS SPIEL GEKAUFT / IN DER BIBLIOTHEK? ---
if ( $is_user_logged_in && $is_in_collection ) {
    // SPIEL GEKAUFT/IN BIBLIOTHEK (Play/Download Buttons)

    $has_play_or_download = false;
    $owned_buttons_html = '';

    // Play Button (Spiel-URL)
    if ( ! empty($game_url) ) {
        $owned_buttons_html .= '<a href="' . esc_url( $game_url ) . '" target="_blank" class="slgs-play-button slgs-is-claimed" data-post-id="' . esc_attr( $post_id ) . '" id="slgs-play-button"><span class="dashicons dashicons-controls-play"></span> ' . __( 'Spielen (In der Bibliothek)', 'steam-like-shop' ) . '</a>';
        $has_play_or_download = true;
    }

    // Download Button (Download-URL) - Zeigt UNABHÄNGIG von der Game URL
    if ( ! empty($download_url) ) {
        $owned_buttons_html .= '<a href="' . esc_url( $download_url ) . '" target="_blank" class="slgs-download-button" download><span class="dashicons dashicons-download"></span> ' . __( 'Herunterladen', 'steam-like-shop' ) . '</a>';
        $has_play_or_download = true;
    }

    if ( $has_play_or_download ) {
        echo $owned_buttons_html;
        // Die Notiz (Spiel startet im Browser) wird hier nur bei Besitzern angezeigt, wenn mindestens ein Link vorhanden ist.
        echo '<p class="slgs-game-note">' . __( 'Dieses Spiel wird direkt in Ihrem Browser gestartet. Es ist keine Installation erforderlich.', 'steam-like-shop' ) . '</p>';
    }

} else {
    // --- 2. LOGIK: SPIEL NICHT GEKAUFT / NICHT IN BIBLIOTHEK (Zeige KAUFBUTTON IMMER) ---

    // Meldung für nicht angemeldete Benutzer
    if ( ! $is_user_logged_in ) {
        $login_url = wp_login_url( get_permalink() );
        echo '<a href="' . esc_url( $login_url ) . '" class="slgs-login-to-buy-button"><span class="dashicons dashicons-lock"></span> ' . __( 'Anmelden zum Kaufen / Beanspruchen', 'steam-like-shop' ) . '</a>';
        echo '<p class="slgs-warning slgs-not-logged-in-msg">' . __( 'Bitte melden Sie sich an, um dieses Spiel zu kaufen oder zu beanspruchen.', 'steam-like-shop' ) . '</p>';

    } elseif ( $is_in_cart ) {
        // Angemeldet, im Warenkorb
        echo '<a href="' . esc_url($checkout_url) . '" class="slgs-go-to-cart-button" id="slgs-go-to-cart-button-link" data-post-id="' . esc_attr( $post_id ) . '" data-price="' . esc_attr($price_float) . '"><span class="dashicons dashicons-cart"></span> ' . __( 'Im Warenkorb (Checkout)', 'steam-like-shop' ) . '</a>';

    } else {
        // Angemeldet, Nicht gekauft, Nicht im Warenkorb (Der eigentliche Kauf-Button)
        $button_text = ($price_float > 0.00) ? __( 'In den Warenkorb (' . $price_display . ')', 'steam-like-shop' ) : __( 'Zum Warenkorb hinzufügen', 'steam-like-shop' );
        $button_class = ($price_float > 0.00) ? 'slgs-add-to-cart-button slgs-buy-button' : 'slgs-add-to-cart-button slgs-claim-button';
        echo '<button class="' . esc_attr($button_class) . '" data-post-id="' . esc_attr( $post_id ) . '" id="slgs-add-to-cart-button" data-price="' . esc_attr($price_float) . '" data-mode="add"><span class="dashicons dashicons-cart"></span> <span class="slgs-cart-button-text">' . esc_html($button_text) . '</span></button>';
    }

    // Die Notiz (Spiel startet im Browser) wird hier für alle Nicht-Besitzer ausgegeben, wenn es einen Play-Link gibt.
    // ACHTUNG: Die ursprüngliche Logik war fehlerhaft und hat die Notiz nur gezeigt, wenn $game_url gesetzt war, aber keinen Download hatte.
    // Mit diesem Patch wird die Notiz konsistent bei Nicht-Besitzern angezeigt, da sie zur Kauf-Aktion gehört.
    echo '<p class="slgs-game-note">' . __( 'Dieses Spiel wird direkt in Ihrem Browser gestartet. Es ist keine Installation erforderlich.', 'steam-like-shop' ) . '</p>';
}

echo '<div id="slgs-action-feedback" class="slgs-action-feedback"></div>';
echo '</div>';

// Der folgende Code für Wunschliste und Teilen bleibt unverändert.

        $wishlist_text = $is_on_wishlist ? __( 'Auf Wunschliste', 'steam-like-shop' ) : __( 'Zur Wunschliste hinzufügen', 'steam-like-shop' );
        $wishlist_class = $is_on_wishlist ? 'slgs-wishlist-added' : 'slgs-wishlist-add';
        $wishlist_disabled = $is_in_collection ? 'disabled' : '';

        echo '<button class="slgs-wishlist-button-new ' . esc_attr( $wishlist_class ) . '" data-post-id="' . esc_attr( $post_id ) . '" data-added-text="' . esc_attr( __( 'Auf Wunschliste', 'steam-like-shop' ) ) . '" data-add-text="' . esc_attr( __( 'Zur Wunschliste hinzufügen', 'steam-like-shop' ) ) . '" ' . esc_attr($wishlist_disabled) . '><span class="dashicons dashicons-heart"></span> <span class="slgs-wishlist-text-new">' . $wishlist_text . '</span></button>';

        echo '<button class="slgs-share-button" id="slgs-share-button"><span class="dashicons dashicons-share"></span> <span class="slgs-share-text">' . __( 'Link teilen', 'steam-like-shop' ) . '</span></button>';

        echo '</div>';
        echo '</div>';

        echo '<div class="slgs-content-tabs-container">';
        echo '<nav class="slgs-content-tab-nav">';
        echo '<button class="slgs-content-tab-button slgs-content-tab-active" data-content-tab="description">' . __( 'Beschreibung', 'steam-like-shop' ) . '</button>';
        echo '<button class="slgs-content-tab-button" data-content-tab="updates">' . __( 'Updates', 'steam-like-shop' ) . '</button>';
        echo '</nav>';

        // HÖHE ANPASSUNG für scrollbaren Inhalt (CSS Korrektur im enqueues)
        echo '<div id="slgs-content-tab-description" class="slgs-content-tab-panel slgs-content-tab-active">';
        echo '<div class="slgs-description-box"><div class="slgs-description-content">' . $content . '</div></div>';
        echo '</div>';

        echo '<div id="slgs-content-tab-updates" class="slgs-content-tab-panel" style="display: none;">';
        echo '<div class="slgs-update-log-box">';
        if ( ! empty( $update_log_raw ) ) {
            echo '<div class="slgs-update-log">';
            $updates = explode("\n", trim($update_log_raw));
            foreach ($updates as $line) {
                $line = trim($line);
                if (empty($line)) continue;
                if (preg_match('/^V(\d+(\.\d+)*):/i', $line)) {
                    $line = str_replace(':', '</strong>:', $line);
                    echo '<h4>' . '<strong>' . $line . '</h4>';
                } else {
                    echo '<p>- ' . esc_html($line) . '</p>';
                }
            }
            echo '</div>';
        } else {
            echo '<p class="slgs-info">' . __( 'Für dieses Spiel sind noch keine Updates veröffentlicht worden.', 'steam-like-shop' ) . '</p>';
        }
        echo '</div>';
        echo '</div>';

        echo '</div>';


        if ( $trailer_url ) {
            echo '<h2>' . __( 'Trailer', 'steam-like-shop' ) . '</h2>';
            echo '<div class="slgs-trailer-embed">';
            $embed = wp_oembed_get( $trailer_url, array( 'width' => 800 ) );
            echo $embed ? $embed : '<p class="slgs-warning">' . __( 'Trailer konnte nicht eingebettet werden (ungültige URL).', 'steam-like-shop' ) . '</p>';
            echo '</div>';
        }

        if ($has_minimum || $has_recommended) {
            echo '<h2>' . __( 'Systemanforderungen', 'steam-like-shop' ) . '</h2>';
            echo '<div class="slgs-requirements-container">';

            echo '<div class="slgs-requirements-box"><h3>' . __( 'Minimal', 'steam-like-shop' ) . '</h3><ul>';
            if ($has_minimum) {
                if ($min_os) echo '<li><strong>' . __( 'Betriebssystem:', 'steam-like-shop' ) . '</strong> ' . esc_html($min_os) . '</li>';
                if ($min_cpu) echo '<li><strong>' . __( 'Prozessor:', 'steam-like-shop' ) . '</strong> ' . esc_html($min_cpu) . '</li>';
                if ($min_ram) echo '<li><strong>' . __( 'Arbeitsspeicher:', 'steam-like-shop' ) . '</strong> ' . esc_html($min_ram) . '</li>';
                if ($min_gpu) echo '<li><strong>' . __( 'Grafikkarte:', 'steam-like-shop' ) . '</strong> ' . esc_html($min_gpu) . '</li>';
            } else { echo '<p class="slgs-warning-small">' . __( 'Minimale Spezifikationen nicht angegeben.', 'steam-like-shop' ) . '</p>'; }
            echo '</ul></div>';

            echo '<div class="slgs-requirements-box slgs-requirements-recommended"><h3>' . __( 'Empfohlen', 'steam-like-shop' ) . '</h3><ul>';
            if ($has_recommended) {
                if ($rec_os) echo '<li><strong>' . __( 'Betriebssystem:', 'steam-like-shop' ) . '</strong> ' . esc_html($rec_os) . '</li>';
                if ($rec_cpu) echo '<li><strong>' . __( 'Prozessor:', 'steam-like-shop' ) . '</strong> ' . esc_html($rec_cpu) . '</li>';
                if ($rec_ram) echo '<li><strong>' . __( 'Arbeitsspeicher:', 'steam-like-shop' ) . '</strong> ' . esc_html($rec_ram) . '</li>';
                if ($rec_gpu) echo '<li><strong>' . __( 'Grafikkarte:', 'steam-like-shop' ) . '</strong> ' . esc_html($rec_gpu) . '</li>';
            } else { echo '<p class="slgs-warning-small">' . __( 'Empfohlene Spezifikationen nicht angegeben.', 'steam-like-shop' ) . '</p>'; }
            echo '</ul></div>';

            echo '</div>';
        }

        echo '<h2>' . __( 'Bewertung & Statistiken', 'steam-like-shop' ) . '</h2>';
        echo '<div class="slgs-stats-container">';

        echo '<div class="slgs-rating-box ' . esc_attr($rating_color_class) . '"><h3>' . __( 'Durchschnittliche Bewertung', 'steam-like-shop' ) . '</h3><p class="slgs-rating-stars" id="slgs-avg-rating-display">' . $rating_display . '</p><p class="slgs-rating-value" id="slgs-avg-value">' . number_format((float)$average_rating, 1, ',', '') . ' / 5.0</p><p class="slgs-rating-count" id="slgs-total-votes">(' . number_format_i18n($total_ratings) . ' ' . __( 'Stimmen', 'steam-like-shop' ) . ')</p></div>';

        // Bewertungs-Kasten nur anzeigen, wenn das Spiel gekauft wurde.
        if ( $is_user_logged_in && $is_in_collection ) {
            echo '<div class="slgs-user-rating-box">';

            echo '<h3>' . __( 'Ihre Bewertung', 'steam-like-shop' ) . '</h3>';
            $status_html = $user_has_rated ? ('Sie haben dieses Spiel mit <span class="slgs-user-rated-value">' . $user_rating . '</span> Sternen bewertet.') : ('Klicken Sie auf einen Stern, um Ihre Bewertung abzugeben.');
            $rated_class = $user_has_rated ? ' slgs-rated' : '';
            echo '<p class="slgs-user-rating-status" id="slgs-user-rating-status">' . __($status_html, 'steam-like-shop') . '</p>';
            echo '<div class="slgs-rating-stars-interactive' . $rated_class . '" data-post-id="' . esc_attr($post_id) . '" data-current-rating="' . esc_attr($user_rating) . '">';
            echo slgs_get_rating_stars_interactive( $user_rating, true );
            echo '</div>';
            echo '<p class="slgs-rating-message" id="slgs-rating-message"></p>';
            echo '</div>';
        }
        // Ende Logik

        echo '<div class="slgs-stats-box"><h3>' . __( 'Aktivierungen', 'steam-like-shop' ) . '</h3><p class="slgs-stat-value" id="slgs-activation-display">' . number_format_i18n( $activation_count ) . '</p><p class="slgs-stat-label">' . __( 'Gesamte Klicks auf Spielen', 'steam-like-shop' ) . '</p></div>';

        echo '</div>';

        echo '</div>';

        return ob_get_clean();
    }
    return $content;
}

add_filter( 'the_content', 'slgs_display_game_content', 20 );

// ===============================================
// 6. CSS & JAVASCRIPT (Styling und AJAX)
// ===============================================

function slgs_enqueue_styles() {
    wp_enqueue_style( 'dashicons' );

    global $post;
    $load_scripts = false;

    // Prüfen, ob wir auf einer Spielseite oder einer Seite mit einem der Shortcodes sind.
    if ( is_singular( 'game' ) || is_post_type_archive( 'game' ) || is_tax( 'game_category' ) ) {
        $load_scripts = true;
    }
    elseif ( is_a( $post, 'WP_Post' ) && (
        has_shortcode( $post->post_content, 'slgs_main_hub' ) || has_shortcode( $post->post_content, 'game_shop' ) ||
        has_shortcode( $post->post_content, 'new_games' ) || has_shortcode( $post->post_content, 'top_games' ) ||
        has_shortcode( $post->post_content, 'my_games' ) || has_shortcode( $post->post_content, 'best_rated_games' ) ||
        has_shortcode( $post->post_content, 'my_wishlist' ) || has_shortcode( $post->post_content, 'latest_reviews' ) ||
        has_shortcode( $post->post_content, 'slgs_cart_checkout' ) || has_shortcode( $post->post_content, 'slgs_cart_button' )
    ) ) {
        $load_scripts = true;
    }

    // Inline-Styles für das Basis-Styling
    $custom_css = "
        /* Allgemeine Farben und Schriftarten */
        body { background-color: #1b2838; color: #c7d5e0; font-family: 'Arial', sans-serif; line-height: 1.6; }
        a { color: #66c0f4; text-decoration: none; transition: color 0.2s; }
        a:hover { color: #fff; }
        h2 { color: #fff; border-bottom: 2px solid #334257; padding-bottom: 10px; margin-top: 25px; font-size: 1.8em; font-weight: 300; }
        .slgs-warning { color: #ffb800; background-color: #3a2e1d; border: 1px solid #ffb800; padding: 10px; border-radius: 4px; }
        .slgs-info { color: #66c0f4; background-color: #1a2a3a; border: 1px solid #334257; padding: 10px; border-radius: 4px; }

        /* --- Full-Width Layout für Spiel-Seiten erzwingen --- */
        body.slgs-full-width .site-main, body.slgs-full-width .content-area, body.slgs-full-width .primary { width: 100% !important; max-width: 100% !important; float: none !important; margin-right: 0 !important; padding-right: 0 !important; box-sizing: border-box; }
        body.slgs-full-width #secondary, body.slgs-full-width .widget-area, body.slgs-full-width .sidebar { display: none !important; }

        /* --- Konsistenter Header (überall) --- */
        .slgs-consistent-header {
            background-color: #1b2838; border-radius: 8px; padding: 20px; box-shadow: 0 0 20px rgba(0, 0, 0, 0.5);
            margin: 20px 0 0 0; position: relative;
        }
        .slgs-header-single, .slgs-header-cart {
            margin-bottom: 20px;
        }

        .slgs-main-hub-header { position: relative; margin-bottom: 20px; padding-right: 150px; }
        .slgs-main-hub-header h2 { margin: 0 0 20px 0; border-bottom: none; font-size: 1.8em; font-weight: bold; }

        /* --- Warenkorb Widget (Icon im Header) --- */
        .slgs-cart-widget { position: absolute; top: 20px; right: 20px; display: flex; align-items: center; padding: 10px 15px; border-radius: 4px; background: #334257; color: #c7d5e0 !important; font-weight: bold; font-size: 0.9em; transition: background 0.2s, color 0.2s, box-shadow 0.2s; z-index: 100; }
        .slgs-cart-widget:hover { background: #4a5c71; color: #fff !important; }
        .slgs-cart-widget .dashicons { font-size: 1.5em; margin-right: 5px; color: #c7d5e0; }
        .slgs-cart-count { background: #1b2838; color: #fff; padding: 2px 7px; border-radius: 50%; margin-left: 5px; line-height: 1; min-width: 20px; text-align: center; transition: background 0.2s; }
        /* Active Status (wenn Items im Warenkorb sind) */
        .slgs-cart-widget.slgs-cart-active { background: #ffb800; color: #1b2838 !important; box-shadow: 0 0 10px rgba(255, 184, 0, 0.5); }
        .slgs-cart-widget.slgs-cart-active .dashicons { color: #1b2838; }
        .slgs-cart-widget.slgs-cart-active .slgs-cart-count { background: #1b2838; color: #ffb800; }

        /* Navigation Style */
        .slgs-tab-nav { display: flex; border-bottom: 2px solid #334257; padding: 0; list-style: none; flex-wrap: wrap; }
        .slgs-tab-button { background: none; border: none; padding: 10px 20px; cursor: pointer; color: #66c0f4; font-size: 1.1em; font-weight: bold; transition: color 0.2s, border-bottom 0.2s, background 0.2s; margin-right: 1px; border-bottom: 2px solid transparent; }
        .slgs-tab-button:hover { color: #fff; }

        /* Aktiver Tab im HUB (AJAX-Schalter) */
        .slgs-tab-nav.slgs-nav-hub .slgs-tab-button.slgs-tab-active { color: #fff; border-bottom: 2px solid #66c0f4; background-color: #2a3a4a; }
        .slgs-tab-nav.slgs-nav-hub .slgs-dropdown-menu .slgs-tab-button.slgs-tab-active { border-bottom: none; } /* Dropdown-Button soll keine eigene Linie haben */

        /* Links auf der Single/Cart Seite (sind Links, keine aktiven Tabs) */
        .slgs-tab-nav.slgs-nav-single .slgs-tab-button,
        .slgs-tab-nav.slgs-nav-cart .slgs-tab-button {
             /* Konsistenter Look, aber kein 'active' State, da sie zur Hauptseite führen */
             background: #1b2838; border: 1px solid #334257; border-radius: 4px; padding: 8px 15px; font-size: 0.95em;
             margin-right: 10px; line-height: 1.5;
        }
        .slgs-tab-nav.slgs-nav-single .slgs-tab-button:hover,
        .slgs-tab-nav.slgs-nav-cart .slgs-tab-button:hover { background: #2a3a4a; color: #fff; border-color: #66c0f4; }

        /* --- Shop Dropdown Menü (NEU) --- */
        .slgs-dropdown-menu { position: relative; display: inline-block; }
        .slgs-dropdown-toggle { display: flex; align-items: center; }
        .slgs-dropdown-toggle .dashicons { margin-left: 5px; font-size: 0.8em; }
        .slgs-dropdown-content {
            display: none;
            position: absolute;
            background-color: #2a3a4a;
            min-width: 200px;
            box-shadow: 0px 8px 16px 0px rgba(0,0,0,0.8);
            z-index: 100;
            top: 100%; /* Unter dem Button */
            left: 0;
            border-radius: 0 0 4px 4px;
            border: 1px solid #334257;
            border-top: none;
        }
        .slgs-dropdown-content .slgs-tab-button {
            display: block; /* Alle Button-Links auf volle Breite */
            width: 100%;
            text-align: left;
            padding: 10px 20px;
            margin: 0;
            border-bottom: 1px solid #334257;
            border-radius: 0;
            background: transparent;
        }
        .slgs-dropdown-content .slgs-tab-button:hover { background: #334257; }
        .slgs-dropdown-content .slgs-tab-button:last-child { border-bottom: none; border-radius: 0 0 4px 4px; }
        .slgs-dropdown-menu:hover .slgs-dropdown-content, .slgs-dropdown-menu.slgs-open .slgs-dropdown-content { display: block; }

        .slgs-tab-panel { min-height: 400px; }
        .slgs-tab-panel h2 { display: none; }
        .slgs-large-warning { padding: 20px; border: 1px solid #ff5757; background: #411d1d; border-radius: 6px; text-align: center; font-size: 1.2em; }

        /* --- Shop Listen-Ansicht --- */
        .slgs-game-list { display: grid; grid-template-columns: repeat(auto-fill, minmax(280px, 1fr)); gap: 20px; margin-top: 20px; }
        .slgs-game-card { border-radius: 8px; overflow: hidden; background-color: #1e2d3e; box-shadow: 0 4px 10px rgba(0, 0, 0, 0.7); transition: transform 0.2s, box-shadow 0.2s, background-color 0.2s; position: relative; }
        .slgs-game-card:hover { transform: translateY(-5px); background-color: #2a3a4a; box-shadow: 0 8px 20px rgba(0, 0, 0, 0.9); }
        .slgs-game-card:hover .slgs-card-remove-icon { opacity: 1; }
        .slgs-card-image { width: 100%; height: 180px; object-fit: cover; display: block; }
        .slgs-card-content { padding: 15px; }
        .slgs-card-title a { color: #fff; font-size: 1.3em; font-weight: bold; }
        .slgs-card-creator { font-size: 0.9em; color: #8899a6; margin-bottom: 5px; }
        .slgs-card-excerpt { font-size: 0.95em; color: #c7d5e0; margin-bottom: 15px; display: -webkit-box; -webkit-line-clamp: 3; -webkit-box-orient: vertical; overflow: hidden; text-overflow: ellipsis; min-height: 60px; }

        .slgs-details-button { display: block; text-align: center; background-color: #66c0f4; color: #1b2838 !important; padding: 10px 15px; border-radius: 4px; text-transform: uppercase; font-weight: bold; font-size: 1.0em; transition: background-color 0.2s, transform 0.1s; }
        .slgs-details-button:hover { background-color: #55aadb; transform: translateY(-1px); }

        .slgs-card-remove-icon { position: absolute; top: 10px; right: 10px; background: rgba(255, 87, 87, 0.9); border: 1px solid #ff5757; color: #fff; padding: 5px; border-radius: 50%; cursor: pointer; font-size: 16px; line-height: 1; z-index: 10; transition: opacity 0.2s, background 0.2s; opacity: 0; }
        .slgs-card-remove-icon:hover { background: #ff5757; opacity: 1; }
        .slgs-card-remove-icon .dashicons { font-size: 18px; height: 18px; width: 18px; display: block; }

        .slgs-card-rating { margin-bottom: 10px; }
        .slgs-card-rating .dashicons { font-size: 1.1em; height: 1.1em; width: 1.1em; color: #ffb800; vertical-align: middle; }

        .slgs-card-price { font-size: 1.2em; font-weight: bold; color: #57cb34; margin: 0 0 15px 0; text-align: right; padding: 5px 0; border-top: 1px solid #334257; }

        /* Feature Icons auf der Karte */
        .slgs-game-features { position: absolute; top: 5px; left: 5px; display: flex; gap: 5px; padding: 5px; }
        .slgs-feature-icon { background: rgba(27, 40, 56, 0.7); border-radius: 4px; padding: 3px; }
        .slgs-feature-icon .dashicons { color: #fff; font-size: 16px; }
        .slgs-fsk-badge { font-weight: bold; font-size: 0.8em; padding: 4px 6px; border-radius: 4px; line-height: 1; align-self: center; border: 1px solid rgba(255, 255, 255, 0.5); min-width: 20px; text-align: center; }
        .slgs-fsk-0 { background: #57cb34; color: #1b2838; border-color: #4da32e; }
        .slgs-fsk-6 { background: #2f9498; color: #fff; border-color: #247578; }
        .slgs-fsk-12 { background: #ffb800; color: #1b2838; border-color: #d19900; }
        .slgs-fsk-16 { background: #ff5757; color: #fff; border-color: #d14747; }
        .slgs-fsk-18 { background: #a20000; color: #fff; border-color: #7a0000; }

        .slgs-filter-container { margin-bottom: 20px; display: flex; flex-wrap: wrap; gap: 10px; }
        .slgs-filter-button { background-color: #334257; color: #fff; border: 1px solid #4a5c71; padding: 8px 15px; border-radius: 6px; cursor: pointer; transition: background-color 0.2s, border-color 0.2s; font-weight: bold; }
        .slgs-filter-button:hover { background-color: #4a5c71; border-color: #66c0f4; }
        .slgs-filter-button.slgs-active { background-color: #66c0f4; color: #1b2838; border-color: #fff; }

        /* --- Detailseite --- */
        .slgs-single-details { position: relative; padding-top: 0; } /* Padding entfernt, da Header jetzt konsistent ist */
        .slgs-top-section { display: flex; gap: 30px; margin-bottom: 30px; padding-top: 10px; }

        /* Warenkorb Widget auf Detailseite/Cart in den Header verschoben */
        .slgs-single-details > .slgs-cart-widget { display: none; }
        .slgs-cart-checkout-container > .slgs-cart-widget { display: none; }

        /* NEU: Galerie/Media Container Styling */
        .slgs-media-container { flex-basis: 65%; max-width: 65%; flex-shrink: 0; }
        .slgs-main-media-viewer { margin-bottom: 15px; }
        .slgs-main-screenshot { width: 100%; height: auto; border-radius: 8px; box-shadow: 0 5px 15px rgba(0, 0, 0, 0.6); display: block; }

        .slgs-media-gallery-thumbs {
            display: flex; gap: 10px; overflow-x: auto; padding: 5px 0;
        }
        .slgs-media-thumb {
            width: 120px; height: 67px; /* 16:9 Verhältnis */
            object-fit: cover; border-radius: 4px;
            cursor: pointer; opacity: 0.6;
            transition: opacity 0.2s, border 0.2s, transform 0.1s;
            border: 2px solid transparent;
            flex-shrink: 0; /* Verhindert das Schrumpfen */
        }
        .slgs-media-thumb:hover { opacity: 1; transform: translateY(-1px); }
        .slgs-media-thumb.slgs-thumb-active { opacity: 1; border-color: #66c0f4; cursor: default; }
        /* Ende NEU Galerie/Media Container Styling */

        .slgs-side-panel { flex-basis: 35%; min-width: 250px; display: flex; flex-direction: column; gap: 20px; }

        .slgs-metadata-box { background: #2a3a4a; padding: 15px; border-radius: 8px; font-size: 0.9em; color: #c7d5e0; }
        .slgs-metadata-box p { margin: 5px 0; color: inherit; }
        .slgs-metadata-box strong { color: #fff; }
        .slgs-metadata-box a { color: #c7d5e0; }
        .slgs-metadata-box a:hover { color: #66c0f4; }

        /* Beschreibung/Update Tabs */
        .slgs-content-tabs-container { margin-bottom: 30px; }
        .slgs-content-tab-nav { display: flex; border-bottom: 2px solid #334257; margin-bottom: 0; list-style: none; padding: 0; }
        .slgs-content-tab-button { background: #1b2838; border: 1px solid #334257; border-bottom: none; padding: 10px 20px; cursor: pointer; color: #66c0f4; font-weight: bold; transition: color 0.2s, background 0.2s; margin-right: 2px; border-radius: 6px 6px 0 0; }
        .slgs-content-tab-button:hover { color: #fff; }
        .slgs-content-tab-button.slgs-content-tab-active { color: #fff; background-color: #2a3a4a; border-color: #334257; z-index: 1; }

        /* FIX FÜR SCROLL-PROBLEM: Erzwingt, dass die Scroll-Steuerung nicht vom Tab-Inhalt übernommen wird */
        .slgs-content-tab-panel {
            background: #2a3a4a; padding: 25px; border-radius: 0 8px 8px 8px; border: 1px solid #334257; box-shadow: 0 2px 10px rgba(0, 0, 0, 0.5); margin-top: -1px;
            max-height: none !important;
            overflow: visible !important;
        }
        .slgs-description-content { line-height: 1.7; color: #c7d5e0; }
        .slgs-description-content p { margin-bottom: 15px; }

        .slgs-update-log { color: #c7d5e0; }
        .slgs-update-log h4 { color: #fff; font-size: 1.1em; margin: 15px 0 5px 0; padding-bottom: 5px; border-bottom: 1px dashed #334257; }
        .slgs-update-log p { margin: 5px 0; padding-left: 10px; }

        .slgs-feature-item { background: #1e2d3e; padding: 6px 12px; margin: 5px; border-radius: 6px; color: #c7d5e0; }
        .slgs-feature-item .dashicons { color: #57cb34; }

        /* Play/Cart/Wishlist Section */
        .slgs-play-section { background: #1e2d3e; border: 1px solid #334257; padding: 20px; border-radius: 8px; text-align: center; }
        .slgs-game-price-large { font-size: 1.6em; color: #c7d5e0; margin: 0 0 15px 0; text-align: left; padding-bottom: 10px; border-bottom: 1px solid #334257; }
        .slgs-game-price-large strong { color: #57cb34; margin-left: 10px; }
        .slgs-not-logged-in-msg { color: #ffb800; border: 1px solid #ffb800; padding: 10px; background: #3a2e1d; border-radius: 4px; }

        /* Spielen Button (Grün) */
        .slgs-play-button { display: inline-flex; align-items: center; background-color: #57cb34; color: #1b2838 !important; font-size: 1.2em; padding: 15px 30px; border-radius: 6px; text-decoration: none; font-weight: bold; transition: background-color 0.2s, transform 0.1s; box-shadow: 0 4px 8px rgba(0, 0, 0, 0.4); width: 100%; justify-content: center; margin-bottom: 10px; }
        .slgs-play-button:hover { background-color: #4da32e; transform: translateY(-2px); }

        /* NEU: Download Button (Sekundär) */
        .slgs-download-button { display: inline-flex; align-items: center; justify-content: center; background-color: #334257; color: #fff !important; font-size: 1.2em; padding: 10px 20px; border-radius: 6px; text-decoration: none; font-weight: bold; transition: background-color 0.2s, transform 0.1s; width: 100%; margin-bottom: 10px; margin-top: 5px; }
        .slgs-download-button:hover { background-color: #4a5c71; transform: translateY(-1px); }
        .slgs-download-button .dashicons { margin-right: 10px; font-size: 1.5em; }
        /* ENDE NEU: Download Button */

        /* Cart/Claim Button (Dynamisch) */
        .slgs-add-to-cart-button, .slgs-go-to-cart-button { display: inline-flex; align-items: center; justify-content: center; border: none; font-size: 1.2em; padding: 15px 30px; border-radius: 6px; font-weight: bold; transition: background-color 0.2s, transform 0.1s; box-shadow: 0 4px 8px rgba(0, 0, 0, 0.4); cursor: pointer; width: 100%; margin-bottom: 10px; text-decoration: none; }
        .slgs-add-to-cart-button:hover, .slgs-go-to-cart-button:hover { transform: translateY(-2px); }
        .slgs-go-to-cart-button { background-color: #334257; color: #fff !important; }
        .slgs-go-to-cart-button:hover { background-color: #4a5c71; color: #fff !important; }
        .slgs-add-to-cart-button.slgs-claim-button { background-color: #66c0f4; color: #1b2838 !important; }
        .slgs-add-to-cart-button.slgs-claim-button:hover { background-color: #55aadb; }
        .slgs-add-to-cart-button.slgs-buy-button { background-color: #ffb800; color: #1b2838 !important; }
        .slgs-add-to-cart-button.slgs-buy-button:hover { background-color: #d19900; }
        .slgs-add-to-cart-button .dashicons, .slgs-go-to-cart-button .dashicons { margin-right: 10px; font-size: 1.5em; }

        .slgs-game-note { color: #8899a6; font-size: 0.85em; margin-top: 10px; }
        .slgs-action-feedback { margin-top: 10px; font-size: 0.9em; color: #ffb800; min-height: 20px; text-align: center; font-weight: bold; }
        .slgs-action-feedback.success { color: #57cb34; }
        .slgs-action-feedback.error { color: #ff5757; }


        /* WUNSCHLISTE BUTTON */
        .slgs-wishlist-button-new { display: flex; align-items: center; justify-content: center; padding: 10px; border-radius: 6px; font-weight: bold; cursor: pointer; transition: background-color 0.2s, color 0.2s, transform 0.1s; border: 1px solid #66c0f4; background: #1b2838; color: #66c0f4; width: 100%; }
        .slgs-wishlist-button-new:hover { background: #334257; color: #fff; transform: translateY(-1px); }
        .slgs-wishlist-button-new[disabled] { cursor: not-allowed; opacity: 0.5; background: #2a3a4a; color: #c7d5e0; border-color: #4a5c71; transform: none; }
        .slgs-wishlist-button-new.slgs-wishlist-added { background: #1e2d3e; border-color: #c7d5e0; color: #c7d5e0; }
        .slgs-wishlist-button-new.slgs-wishlist-added:hover { background: #2a3a4a; }
        .slgs-wishlist-button-new .dashicons { margin-right: 5px; }


        /* Share Button */
        .slgs-share-button { display: flex; align-items: center; justify-content: center; padding: 10px; border-radius: 6px; font-weight: bold; cursor: pointer; transition: background-color 0.2s, color 0.2s, transform 0.1s; border: 1px solid #c7d5e0; background: #1b2838; color: #c7d5e0; margin-top: 10px; width: 100%; }
        .slgs-share-button:hover { background: #334257; color: #fff; border-color: #fff; transform: translateY(-1px); }
        .slgs-share-button .dashicons { margin-right: 5px; }

        /* Systemanforderungen */
        .slgs-requirements-container { display: flex; gap: 20px; margin-bottom: 30px; }
        .slgs-requirements-box { flex: 1; background: #1e2d3e; padding: 20px; border-radius: 8px; border: 1px solid #334257; }
        .slgs-requirements-box h3 { color: #fff; border-bottom: 1px solid #334257; padding-bottom: 10px; margin: 0 0 15px 0; font-size: 1.3em; font-weight: bold; }
        .slgs-requirements-box strong { display: inline-block; min-width: 120px; color: #fff; }
        .slgs-requirements-recommended { border: 1px solid #66c0f4; }
        .slgs-requirements-box ul { list-style: none; padding: 0; margin: 0; }
        .slgs-requirements-box li { padding: 5px 0; border-bottom: 1px dotted #334257; color: #c7d5e0; }
        .slgs-requirements-box li:last-child { border-bottom: none; }


        /* Statistiken und Bewertung */
        .slgs-stats-container { display: flex; flex-wrap: wrap; gap: 20px; }
        .slgs-rating-box, .slgs-stats-box, .slgs-user-rating-box { flex: 1; min-width: 200px; padding: 25px; border-radius: 8px; text-align: center; }
        .slgs-user-rating-box { max-width: 33%; background: #2a3a4a; border: 1px solid #334257; } /* Benutzer-Rating-Box bleibt neutral */
        .slgs-rating-box h3, .slgs-stats-box h3, .slgs-user-rating-box h3 { font-size: 1.2em; margin-top: 0; color: #fff; }
        .slgs-rating-value { font-size: 2.5em; font-weight: bold; margin: 10px 0; }
        .slgs-rating-count { color: #8899a6; font-size: 0.9em; }

        /* --- Farb-Codierung für Bewertungen (Grün nach Rot) --- */
        .slgs-rating-box.slgs-rating-excellent { background: #1a4a2a; border: 1px solid #57cb34; }
        .slgs-rating-box.slgs-rating-excellent .slgs-rating-value { color: #57cb34; }

        .slgs-rating-box.slgs-rating-good { background: #2a3a1a; border: 1px solid #ffb800; }
        .slgs-rating-box.slgs-rating-good .slgs-rating-value { color: #ffb800; }

        .slgs-rating-box.slgs-rating-mixed { background: #3a2a1a; border: 1px solid #c7d5e0; }
        .slgs-rating-box.slgs-rating-mixed .slgs-rating-value { color: #c7d5e0; }

        .slgs-rating-box.slgs-rating-bad { background: #4a1a1a; border: 1px solid #ff5757; }
        .slgs-rating-box.slgs-rating-bad .slgs-rating-value { color: #ff5757; }

        .slgs-rating-box.slgs-rating-none { background: #2a3a4a; border: 1px solid #334257; }
        .slgs-stats-box { background: #2a3a4a; border: 1px solid #334257; } /* Stats bleibt neutral */
        /* Ende Farb-Codierung */

        /* Interaktive Sterne */
        .slgs-rating-stars-interactive { display: inline-block; cursor: pointer; white-space: nowrap; }
        .slgs-star-interactive { font-size: 2em; color: #ffb800; transition: color 0.1s; }
        .slgs-star-interactive.dashicons-star-empty { color: #334257; }
        .slgs-user-rating-status { margin-bottom: 15px; color: #c7d5e0; }
        .slgs-user-rated-value { color: #57cb34; font-weight: bold; }

        .slgs-stat-value { font-size: 3em; font-weight: bold; color: #66c0f4; margin: 5px 0 0; }
        .slgs-stat-label { font-size: 0.9em; color: #8899a6; }

        /* --- Warenkorb Styling --- */
        .slgs-cart-checkout-container { background-color: #1b2838; border-radius: 8px; padding: 0 25px 25px 25px; box-shadow: 0 0 20px rgba(0, 0, 0, 0.5); margin: 20px 0; position: relative; }
        .slgs-cart-checkout-container .slgs-consistent-header { margin: 0 -25px 20px -25px; padding: 25px 25px 20px 25px; border-radius: 8px 8px 0 0; }
        .slgs-cart-checkout-container .slgs-main-hub-header h2 { margin-bottom: 0; }

        .slgs-cart-items-list { margin-top: 20px; border-bottom: 1px solid #334257; padding-bottom: 20px; }
        .slgs-cart-item { display: flex; align-items: center; background: #2a3a4a; padding: 15px; border-radius: 6px; margin-bottom: 10px; transition: background 0.2s; }
        .slgs-cart-item:hover { background: #334257; }
        .slgs-cart-thumb { width: 80px; height: 80px; object-fit: cover; border-radius: 4px; margin-right: 15px; flex-shrink: 0; }
        .slgs-cart-item-info { flex-grow: 1; }
        .slgs-cart-item-title { color: #fff; font-weight: bold; font-size: 1.1em; display: block; margin-bottom: 5px; }
        .slgs-cart-item-price { font-weight: bold; font-size: 1.2em; }
        .slgs-cart-item-price.slgs-free { color: #57cb34; }
        .slgs-cart-item-price.slgs-paid { color: #ffb800; }
        .slgs-cart-item-remove { background: #ff5757; border: none; color: #1b2838; cursor: pointer; padding: 8px 10px; margin-left: 15px; border-radius: 4px; transition: background 0.2s; }
        .slgs-cart-item-remove:hover { background: #a20000; color: #fff; }
        .slgs-cart-item-remove .dashicons { font-size: 1.2em; }

        .slgs-cart-summary { text-align: right; margin-top: 20px; padding-top: 15px; border-top: 1px solid #334257; }
        .slgs-cart-total { font-size: 1.6em; color: #fff; margin-bottom: 15px; }
        .slgs-total-cost-value { color: #57cb34; }

        .slgs-checkout-button { display: inline-flex; align-items: center; justify-content: center; width: 100%; max-width: 400px; padding: 15px 30px; border: none; border-radius: 6px; font-weight: bold; font-size: 1.2em; cursor: pointer; transition: background-color 0.2s, transform 0.1s; box-shadow: 0 4px 8px rgba(0, 0, 0, 0.4); }
        .slgs-checkout-button:hover:not([disabled]) { transform: translateY(-2px); }
        .slgs-checkout-button .dashicons { margin-right: 10px; }
        .slgs-checkout-free { background-color: #57cb34; color: #1b2838; }
        .slgs-checkout-free:hover:not([disabled]) { background-color: #4da32e; }
        .slgs-checkout-paid { background-color: #ffb800; color: #1b2838; }
        .slgs-checkout-paid.slgs-disabled { background-color: #4a5c71; color: #c7d5e0; cursor: not-allowed; opacity: 0.7; transform: none; box-shadow: none; }
        .slgs-checkout-message { margin-top: 10px; font-weight: bold; text-align: center; }

        /* Checkout Aktionen ('Einkauf fortsetzen' links, Summary rechts) */
        .slgs-checkout-actions { margin-top: 20px; padding-bottom: 20px; border-bottom: 1px solid #334257; }
        .slgs-checkout-empty { border-bottom: none; } /* Zentriert wenn leer */

        .slgs-continue-shopping-button { display: inline-flex; align-items: center; background-color: #334257; color: #fff !important; padding: 10px 15px; border-radius: 4px; font-weight: bold; transition: background-color 0.2s; }
        .slgs-continue-shopping-button:hover { background-color: #4a5c71; }
        .slgs-continue-shopping-button .dashicons { margin-right: 5px; }

        /* Grid Layout für Warenkorb auf großen Screens */
        @media (min-width: 769px) {
            .slgs-cart-checkout-container {
                display: grid;
                grid-template-areas:
                    'header header'
                    'list list'
                    'actions summary';
                grid-template-columns: 1fr auto;
                gap: 20px;
                padding-top: 25px;
            }
            .slgs-cart-checkout-container .slgs-consistent-header { grid-area: header; padding-top: 25px; }
            .slgs-cart-checkout-container .slgs-main-hub-header { padding-right: 150px; }
            .slgs-cart-items-list { grid-area: list; }
            .slgs-checkout-actions { grid-area: actions; align-self: flex-start; margin-top: 0; padding-top: 0; border-bottom: none; }
            .slgs-cart-summary { grid-area: summary; align-self: flex-start; padding-top: 0; }
            .slgs-checkout-empty { grid-area: actions / list; justify-content: flex-start; }
            .slgs-cart-checkout-container .slgs-cart-summary { width: auto; }
        }

        /* Responsive Anpassungen */
        @media (max-width: 900px) {
            .slgs-top-section { flex-direction: column; }
            .slgs-media-container { max-width: 100%; flex-basis: 100%; }
            .slgs-side-panel { min-width: 100%; }
            .slgs-stats-container { flex-direction: column; }
            .slgs-user-rating-box { max-width: 100%; }
            .slgs-main-hub-header { padding-right: 0; }
            /* Mobile: Warenkorb-Widget unter dem Titel */
            .slgs-consistent-header .slgs-cart-widget {
                position: static; margin-bottom: 10px; display: inline-flex;
            }
            .slgs-main-hub-header { display: flex; flex-direction: column-reverse; }
            .slgs-main-hub-header h2 { margin-bottom: 10px; }


            .slgs-game-features { display: none; }
            .slgs-checkout-button { max-width: 100%; }
            .slgs-checkout-actions { justify-content: center; }
            .slgs-cart-summary { text-align: center; }
            .slgs-checkout-button { margin-left: auto; margin-right: auto; }
            .slgs-cart-checkout-container .slgs-consistent-header { margin: 0 -25px 20px -25px; padding: 15px 25px 15px 25px; }
            .slgs-cart-checkout-container .slgs-main-hub-header { padding-right: 0; }
        }
        @media (max-width: 768px) {
            .slgs-requirements-container { flex-direction: column; }
        }
    ";
    wp_add_inline_style( 'dashicons', $custom_css );

    if ($load_scripts) {
        // Registriere ein Handle für das Skript (ohne externe Datei)
        wp_register_script( 'slgs-frontend-logic', '', array( 'jquery' ), '3.1.23', true );
        wp_enqueue_script( 'slgs-frontend-logic' );

        $checkout_slug = get_option( 'slgs_checkout_page_slug', 'checkout' );
        wp_localize_script( 'slgs-frontend-logic', 'slgs_ajax', array(
            'ajax_url'          => admin_url( 'admin-ajax.php' ),
            'activation_nonce'  => wp_create_nonce( 'slgs_activation_nonce' ),
            'collection_nonce'  => wp_create_nonce( 'slgs_collection_nonce' ),
            'wishlist_nonce'    => wp_create_nonce( 'slgs_wishlist_nonce' ),
            'cart_nonce'        => wp_create_nonce( 'slgs_cart_nonce' ),
            'checkout_nonce'    => wp_create_nonce( 'slgs_checkout_nonce' ),
            'rating_nonce'      => wp_create_nonce( 'slgs_rating_nonce' ),
            'is_logged_in'      => is_user_logged_in(),
            'not_logged_in_msg' => __( 'Bitte melden Sie sich an, um Spiele zu speichern.', 'steam-like-shop' ),
            'empty_lib_msg'     => __( 'Ihre Bibliothek ist leer. Kaufen oder beanspruchen Sie Spiele im Store!', 'steam-like-shop' ),
            'empty_wish_msg'    => __( 'Ihre Wunschliste ist leer.', 'steam-like-shop' ),
            'empty_cart_msg'    => __( 'Ihr Warenkorb ist leer.', 'steam-like-shop' ),
            'currency_symbol'   => get_option( 'slgs_currency_symbol', '€' ),
            'checkout_url'      => home_url( '/' . $checkout_slug ),
            'initial_cart_count' => slgs_get_user_cart_count(),
        ) );

        // Inline-Skript für AJAX und Filter (Frontend)
        $inline_script = "
            jQuery(document).ready(function(\$) {

                // --- 0. HILFSFUNKTIONEN FÜR RATING UND CART DISPLAY ---
                function updateRatingStars(container, rating) {
                    container.find('.dashicons').each(function() {
                        var starValue = \$(this).data('rating');
                        var isFilled = starValue <= rating;
                        \$(this).toggleClass('dashicons-star-filled', isFilled).toggleClass('dashicons-star-empty', !isFilled);
                    });
                }

                function updateCartStatus(\$buttonOrLink, action, post_id) {
                    var addedText = 'Im Warenkorb (Checkout)';
                    var priceFloat = parseFloat(\$buttonOrLink.data('price'));
                    var priceText = priceFloat.toFixed(2).replace('.', ',') + ' ' + slgs_ajax.currency_symbol;
                    var isBuyButton = priceFloat > 0;
                    var addText = isBuyButton ? \"In den Warenkorb (\" + priceText + \")\" : \"Zum Warenkorb hinzufügen\";

                    if (action === 'added') {
                        var \$link = \$('<a>').attr('href', slgs_ajax.checkout_url).addClass('slgs-go-to-cart-button').attr('id', 'slgs-go-to-cart-button-link').attr('data-post-id', post_id).attr('data-price', priceFloat).html('<span class=\"dashicons dashicons-cart\"></span> <span class=\"slgs-cart-button-text\">' + addedText + '</span>');
                        \$buttonOrLink.replaceWith(\$link);
                    } else if (action === 'removed') {
                        var \$newButton = \$('<button>').addClass('slgs-add-to-cart-button ' + (isBuyButton ? 'slgs-buy-button' : 'slgs-claim-button')).attr('id', 'slgs-add-to-cart-button').attr('data-post-id', post_id).attr('data-price', priceFloat).attr('data-mode', 'add').html('<span class=\"dashicons dashicons-cart\"></span> <span class=\"slgs-cart-button-text\">' + addText + '</span>');
                        \$buttonOrLink.replaceWith(\$newButton);
                        \$newButton.prop('disabled', false);
                    }
                    // Wichtig: Beim Wechsel von Cart zu Play Button oder umgekehrt, muss der Download-Button (falls vorhanden) verschwinden/erscheinen.
                    if (action === 'removed') { // Wenn Spiel aus der Bib entfernt, Download verschwindet
                        \$('.slgs-download-button').remove();
                    }
                }

                // nderung: Widget wird immer angezeigt, nur Active Class wird entfernt/hinzugefügt
                function updateCartWidget(newCount) {
                    var \$widget = \$('#slgs-cart-widget');
                    \$widget.find('.slgs-cart-count').text(newCount);
                    \$widget.toggleClass('slgs-cart-active', newCount > 0);
                    // Sicherstellen, dass das Widget sichtbar ist
                    \$widget.show();
                }

                if (\$('#slgs-cart-widget').length && slgs_ajax.initial_cart_count !== undefined) {
                    updateCartWidget(slgs_ajax.initial_cart_count);
                }


                // --- 1. SHOP FILTER LOGIK (Jetzt über Dropdown gesteuert) ---
                // Alter Handler für die Filterleiste wurde entfernt

// --- 2. AKTIVIERUNGS-TRACKING (Klick auf Spielen) ---
   \$('body').on('click', '.slgs-play-button', function(e) {
                    var \$button = \$(this);
                    var post_id = \$button.data('post-id');

                    if (post_id) {
                        \$.ajax({
                            url: slgs_ajax.ajax_url,
                            type: 'POST',
                            data: {
                                action: 'slgs_track_activation',
                                post_id: post_id,
                                nonce: slgs_ajax.activation_nonce
                            },
                            success: function(response) {
                                if (response.success) {
                                    console.log('Aktivierung gezhlt: ' + response.data.new_count);
                                }
                            },
                            error: function() {
                                console.log('Fehler beim Tracking der Aktivierung');
                            }
                        });
                    }
                });

                // --- 3. WARENKORB LOGIK (Add to Cart) ---
                \$('body').on('click', '#slgs-add-to-cart-button', function() {
                    if (!slgs_ajax.is_logged_in) { console.log(slgs_ajax.not_logged_in_msg); return; }

                    var \$button = \$(this);
                    var post_id = \$button.data('post-id');
                    var mode = \$button.data('mode');
                    var \$feedback = \$('#slgs-action-feedback');
                    var originalHTML = \$button.html(); \$button.data('original-html', originalHTML);

                    \$button.prop('disabled', true).html('<span class=\"dashicons dashicons-update-alt spin\"></span> Wird hinzugefügt...');
                    \$feedback.text('').removeClass('success error');

                    \$.ajax({
                        url: slgs_ajax.ajax_url, type: 'POST',
                        data: { action: 'slgs_toggle_cart', post_id: post_id, nonce: slgs_ajax.cart_nonce, mode: mode },
                        success: function(response) {
                            var restoreHTML = \$button.data('original-html') || originalHTML;
                            \$button.prop('disabled', false).html(restoreHTML);

                            if (response.success && response.data.action === 'added') {
                                \$feedback.addClass('success').text('Spiel zum Warenkorb hinzugefügt!');
                                if (\$button.closest('.slgs-single-details').length) { updateCartStatus(\$button, 'added', post_id); }
                                updateCartWidget(response.data.cart_count);
                            } else {
                                var errorMsg = response.data.message || 'Fehler beim Hinzufügen.';
                                \$feedback.addClass('error').text(errorMsg);
                                console.error('Fehler beim Hinzufügen zum Warenkorb: ' + errorMsg);
                            }
                        },
                        error: function() {
                             var restoreHTML = \$button.data('original-html') || originalHTML;
                             \$button.prop('disabled', false).html(restoreHTML);
                             \$feedback.addClass('error').text(\"Netzwerkfehler: Die Aktion konnte nicht ausgeführt werden.\");
                        }
                    });
                });

                function updateCartTotal(priceChange) {
                    var \$totalCost = \$('#slgs-total-cost');
                    var currentTotalText = \$totalCost.text().replace(slgs_ajax.currency_symbol, '').trim();
                    // Entferne Tausender-Trennzeichen (. ) und ersetze Komma durch Punkt für parseFloat
                    var currentTotal = parseFloat(currentTotalText.replace(/\\./g, '').replace(',', '.'));
                    if (isNaN(currentTotal)) { currentTotal = 0.00; }
                    var newTotal = currentTotal + priceChange;
                    // Formatierung zurück: Punkt als Tausender-Trennzeichen, Komma als Dezimal-Trennzeichen
                    var newTotalDisplay = newTotal.toFixed(2).replace('.', ',').replace(/\\B(?=(\\d{3})+(?!\\d))/g, '.') + ' ' + slgs_ajax.currency_symbol;

                    \$totalCost.text(newTotalDisplay);

                    var \$checkoutButton = \$('#slgs-checkout-button');
                    var \$checkoutText = \$('#slgs-checkout-text');
                    var \$warningSmall = \$('.slgs-warning-small');

                    var hasItems = \$('.slgs-cart-item').length > 0;

                    if (!hasItems) {
                        \$checkoutButton.prop('disabled', true).removeClass('slgs-checkout-free slgs-checkout-paid slgs-disabled');
                        \$checkoutText.text(\"Warenkorb leer\");
                        \$warningSmall.hide();
                        \$('.slgs-cart-summary').hide();
                        // Zeigt NUR den \"Einkauf fortsetzen\" Button an
                        \$('.slgs-checkout-actions').show().addClass('slgs-checkout-empty');
                    } else if (newTotal > 0.00) {
                        \$checkoutButton.prop('disabled', true).removeClass('slgs-checkout-free').addClass('slgs-checkout-paid slgs-disabled');
                        \$checkoutText.text(\"Kauf abschließen (Zahlung erforderlich)\");
                        \$warningSmall.show();
                        \$('.slgs-cart-summary').show();
                        \$('.slgs-checkout-actions').show().removeClass('slgs-checkout-empty');
                    } else if (newTotal === 0.00) {
                        \$checkoutButton.prop('disabled', false).removeClass('slgs-checkout-paid slgs-disabled').addClass('slgs-checkout-free');
                        \$checkoutText.text(\"Kostenlose Spiele beanspruchen (0,00 \" + slgs_ajax.currency_symbol + \")\");
                        \$warningSmall.hide();
                        \$('.slgs-cart-summary').show();
                        \$('.slgs-checkout-actions').show().removeClass('slgs-checkout-empty');
                    }
                }

                // --- 3.2 WARENKORB ENTFERNEN (Handler) ---
                \$('body').on('click', '.slgs-cart-item-remove, .slgs-card-remove-icon[data-list-type=\"cart\"]', function(e) {
                    e.preventDefault(); e.stopPropagation();
                    if (!slgs_ajax.is_logged_in) { return; }
                    var \$button = \$(this);
                    var post_id = \$button.data('post-id');
                    var list_type = \$button.data('list-type') || 'cart';
                    var is_checkout_page = list_type === 'cart';
                    var item_price = is_checkout_page ? parseFloat(\$button.closest('.slgs-cart-item').data('price')) : 0;

                    var \$icon = \$button.find('.dashicons');
                    var original_icon_class = \$icon.attr('class');
                    \$icon.attr('class', 'dashicons dashicons-update-alt spin');
                    \$button.prop('disabled', true);


                    \$.ajax({
                        url: slgs_ajax.ajax_url, type: 'POST',
                        data: { action: 'slgs_toggle_cart', post_id: post_id, nonce: slgs_ajax.cart_nonce, mode: 'remove' },
                        success: function(response) {
                            if (response.success && response.data.action === 'removed') {
                                updateCartWidget(response.data.cart_count);

                                \$button.closest((is_checkout_page ? '.slgs-cart-item' : '.slgs-game-card')).fadeOut(300, function() {
                                    \$(this).remove();

                                    if (!is_checkout_page && \$('.slgs-single-details').length) {
                                         var \$link = \$('#slgs-go-to-cart-button-link[data-post-id=\"' + post_id + '\"]');
                                         if (\$link.length) { updateCartStatus(\$link, 'removed', post_id); }
                                    }

                                    if (is_checkout_page) {
                                         updateCartTotal(-item_price);
                                         if (\$('.slgs-cart-item').length === 0) {
                                             \$('#slgs-cart-items-list').html('<p class=\"slgs-info\">' + slgs_ajax.empty_cart_msg + '</p>');
                                             // updateCartTotal managed the rest
                                         }
                                    }

                                    // Handle empty list in hub tabs
                                    var \$list = \$('.slgs-list-' + list_type);
                                    if (\$list.length && \$list.find('.slgs-game-card').length === 0) {
                                        setTimeout(function() {
                                            if (\$list.find('.slgs-game-card').length === 0) {
                                                var empty_msg = (list_type === 'my_games') ? slgs_ajax.empty_lib_msg : slgs_ajax.empty_wish_msg;
                                                \$list.append('<p class=\"slgs-info\">' + empty_msg + '</p>');
                                            }
                                        }, 100);
                                    }
                                });
                            } else {
                                \$icon.attr('class', original_icon_class); \$button.prop('disabled', false);
                            }
                        },
                        error: function() {
                            \$icon.attr('class', original_icon_class); \$button.prop('disabled', false);
                        }
                    });
                });

                // --- 3.3 CHECKOUT BUTTON KLICK ---
                \$('body').on('click', '#slgs-checkout-button:not([disabled])', function() {
                    if (!slgs_ajax.is_logged_in) { return; }

                    var \$button = \$(this);
                    var nonce = \$button.data('nonce');
                    var \$message = \$('#slgs-checkout-message');
                    var originalText = \$button.find('#slgs-checkout-text').text();

                    \$button.prop('disabled', true).find('#slgs-checkout-text').html('<span class=\"dashicons dashicons-update-alt spin\"></span> Wird verarbeitet...');
                    \$message.text('').removeClass('success error');

                    \$.ajax({
                        url: slgs_ajax.ajax_url, type: 'POST',
                        data: { action: 'slgs_process_checkout', nonce: nonce },
                        success: function(response) {
                            if (response.success) {
                                \$message.css('color', '#57cb34').text(response.data.message).addClass('success');

                                // Reset cart view
                                \$('#slgs-cart-items-list').html('<p class=\"slgs-info\">' + slgs_ajax.empty_cart_msg + '</p>');
                                // updateCartTotal handles visibility
                                updateCartTotal(-10000000); // Trigger update to show empty state correctly
                                \$button.prop('disabled', true).find('#slgs-checkout-text').text('Erfolgreich beansprucht!');

                                // Update detail pages buttons
                                response.data.purchased_ids.forEach(function(id) {
                                    var \$target = \$('#slgs-add-to-cart-button[data-post-id=\"' + id + '\"]');
                                    var \$targetLink = \$('#slgs-go-to-cart-button-link[data-post-id=\"' + id + '\"]');

                                    var \$replaceTarget = (\$target.length) ? \$target : \$targetLink;

                                    if (\$replaceTarget.length) {
                                        var gameUrl = \$replaceTarget.closest('.slgs-single-details').find('.slgs-play-button').attr('href') || '#';
                                        var \$playButton = \$('<a>').attr('href', gameUrl).attr('target', '_blank').addClass('slgs-play-button slgs-is-claimed').attr('data-post-id', id).attr('id', 'slgs-play-button').html('<span class=\"dashicons dashicons-controls-play\"></span> Spielen (In der Bibliothek)');
                                        \$replaceTarget.replaceWith(\$playButton);
                                        \$('.slgs-wishlist-button-new[data-post-id=\"' + id + '\"]').prop('disabled', true);
                                    }
                                });
                            } else {
                                var errorMsg = response.data.message || 'Ein unbekannter Serverfehler ist aufgetreten.';
                                \$message.css('color', '#ff5757').text('Fehler: ' + errorMsg).addClass('error');
                                \$button.prop('disabled', false).find('#slgs-checkout-text').text(originalText);
                            }
                        },
                        error: function() {
                             \$message.css('color', '#ff5757').text('Netzwerkfehler: Bitte versuchen Sie es erneut.').addClass('error');
                             \$button.prop('disabled', false).find('#slgs-checkout-text').text(originalText);
                        }
                    });
                });

                // --- 3.4 ENTFERNEN aus BIBLIOTHEK / WUNSCHLISTE (Icon Click Handler) ---
                \$('body').on('click', '.slgs-card-remove-icon:not([data-list-type=\"cart\"])', function(e) {
                    e.preventDefault(); e.stopPropagation();
                    if (!slgs_ajax.is_logged_in) { return; }
                    var \$button = \$(this);
                    var post_id = \$button.data('post-id');
                    var action_name = \$button.data('action');
                    var nonce_key_name = \$button.data('nonce-key');
                    var list_type = \$button.data('list-type');

                    var \$icon = \$button.find('.dashicons');
                    var original_icon_class = \$icon.attr('class');
                    \$icon.attr('class', 'dashicons dashicons-update-alt spin');
                    \$button.prop('disabled', true);


                    \$.ajax({
                        url: slgs_ajax.ajax_url, type: 'POST',
                        data: { action: action_name, post_id: post_id, nonce: slgs_ajax[nonce_key_name], mode: 'remove' },
                        success: function(response) {
                            \$icon.attr('class', original_icon_class); \$button.prop('disabled', false);

                            if (response.success && response.data.action === 'removed') {
                                if (list_type === 'my_games') {
                                    // Update button on detail page
                                    var \$playButton = \$('#slgs-play-button[data-post-id=\"' + post_id + '\"]');
                                    if (\$playButton.length) { updateCartStatus(\$playButton, 'removed', post_id); }
                                    // Re-enable wishlist button
                                    \$('.slgs-wishlist-button-new[data-post-id=\"' + post_id + '\"]').prop('disabled', false).removeClass('slgs-wishlist-added').addClass('slgs-wishlist-add').find('.slgs-wishlist-text-new').text(\$('.slgs-wishlist-button-new').data('add-text'));
                                    // Hide download button
                                    \$('.slgs-download-button').remove();
                                }

                                if (list_type === 'wishlist' || list_type === 'my_wishlist') {
                                    // Update button on detail page
                                    \$('.slgs-wishlist-button-new[data-post-id=\"' + post_id + '\"]').removeClass('slgs-wishlist-added').addClass('slgs-wishlist-add').find('.slgs-wishlist-text-new').text(\$('.slgs-wishlist-button-new').data('add-text'));
                                }


                                \$button.closest('.slgs-game-card').fadeOut(300, function() {
                                    \$(this).remove();
                                    var \$list = \$('.slgs-list-' + list_type);
                                    if (\$list.length && \$list.find('.slgs-game-card:visible').length === 0) {
                                         setTimeout(function() {
                                            if (\$list.find('.slgs-game-card').length === 0) {
                                                var empty_msg = (list_type === 'my_games') ? slgs_ajax.empty_lib_msg : slgs_ajax.empty_wish_msg;
                                                \$list.append('<p class=\"slgs-info\">' + empty_msg + '</p>');
                                            }
                                        }, 100);
                                    }
                                });
                            } else {
                                console.error('Fehler beim Entfernen: ' + (response.data.message || 'Unbekannt.'));
                            }
                        },
                        error: function() {
                            \$icon.attr('class', original_icon_class); \$button.prop('disabled', false);
                            console.error('Es ist ein unbekannter Fehler aufgetreten.');
                        }
                    });
                });

                // CSS für den Spin-Effekt
                if (!\$('#slgs-spin-style').length) {
                    \$('head').append('<style id=\"slgs-spin-style\">.spin { animation: slgs-spin 1s infinite linear; } @keyframes slgs-spin { from { transform: rotate(0deg); } to { transform: rotate(359deg); } }</style>');
                }

                // --- 3.5 WUNSCHLISTE LOGIK (Toggle) ---
                \$('body').on('click', '.slgs-wishlist-button-new:not([disabled])', function() {
                    if (!slgs_ajax.is_logged_in) { return; }
                    var \$button = \$(this);
                    var post_id = \$button.data('post-id');
                    var is_added = \$button.hasClass('slgs-wishlist-added');

                    \$button.prop('disabled', true).find('.slgs-wishlist-text-new').text('...');
                    var mode = is_added ? 'remove' : 'add';

                    \$.ajax({
                        url: slgs_ajax.ajax_url, type: 'POST',
                        data: { action: 'slgs_toggle_wishlist', post_id: post_id, nonce: slgs_ajax.wishlist_nonce, mode: mode },
                        success: function(response) {
                            \$button.prop('disabled', false);
                            if (response.success) {
                                if (response.data.action === 'added') {
                                    \$button.removeClass('slgs-wishlist-add').addClass('slgs-wishlist-added');
                                    \$button.find('.slgs-wishlist-text-new').text(\$button.data('added-text'));
                                } else {
                                    \$button.removeClass('slgs-wishlist-added').addClass('slgs-wishlist-add');
                                    \$button.find('.slgs-wishlist-text-new').text(\$button.data('add-text'));
                                }
                            } else {
                                \$button.find('.slgs-wishlist-text-new').text('Fehler');
                            }
                        }
                    });
                });

                // --- 4. TAB SWITCHING LOGIK (Hub) ---
                function switchTab(tab_key) {
                    \$('.slgs-tab-button').removeClass('slgs-tab-active');
                    \$('.slgs-main-hub').find('.slgs-tab-button[data-tab=\"' + tab_key + '\"]').addClass('slgs-tab-active');
                    \$('.slgs-tab-panel').hide();
                    var \$targetPanel = \$('#slgs-tab-' + tab_key);
                    \$targetPanel.fadeIn(300);

                    // Resetzt den Filter beim Wechsel
                    if (tab_key === 'shop') {
                        \$targetPanel.find('.slgs-game-card').show();
                        \$targetPanel.find('.slgs-filter-container').hide(); // Filter versteckt lassen
                    }
                }

                \$('.slgs-main-hub').on('click', '.slgs-tab-button:not(.slgs-category-link):not(.slgs-dropdown-toggle)', function() {
                    var tab_key = \$(this).data('tab');
                    switchTab(tab_key);
                    // Sicherstellen, dass der Dropdown-Status zurückgesetzt wird
                    \$('.slgs-dropdown-menu').removeClass('slgs-open').find('.slgs-dropdown-content').hide();
                });

                // NEU: Dropdown Kategorien Klick-Handler
                \$('.slgs-main-hub').on('click', '.slgs-category-link', function(e) {
                    e.preventDefault();
                    var term = \$(this).data('term');
                    var \$shopPanel = \$('#slgs-tab-shop');

                    // Tab wechseln (Store-Tab aktivieren)
                    switchTab('shop');
                    \$('.slgs-dropdown-toggle').addClass('slgs-tab-active');
                    \$('.slgs-category-link').removeClass('slgs-tab-active');
                    \$(this).addClass('slgs-tab-active');

                    // Filter-Logik anwenden
                    var \$gameList = \$shopPanel.find('.slgs-game-list');

                    if (term === 'all') {
                        \$gameList.find('.slgs-game-card').fadeIn(200);
                    } else {
                        \$gameList.find('.slgs-game-card').hide();
                        \$gameList.find('.' + term).fadeIn(200);
                    }

                    // Dropdown schließen
                    \$(this).closest('.slgs-dropdown-content').hide();
                    \$(this).closest('.slgs-dropdown-menu').removeClass('slgs-open');
                });

                // Dropdown Toggle (damit es auf Mobile per Click funktioniert)
                \$('.slgs-main-hub').on('click', '.slgs-dropdown-toggle', function(e) {
                    var \$menu = \$(this).closest('.slgs-dropdown-menu');
                    if (!\$menu.hasClass('slgs-open')) {
                        e.preventDefault();
                        \$('.slgs-dropdown-menu').removeClass('slgs-open').find('.slgs-dropdown-content').hide();
                        \$menu.addClass('slgs-open').find('.slgs-dropdown-content').show();
                    } else {
                        // Wenn es bereits offen ist, führt es die normale Tab-Logik aus (all games)
                        var tab_key = \$(this).data('tab');
                        switchTab(tab_key);
                        \$('.slgs-dropdown-menu').removeClass('slgs-open').find('.slgs-dropdown-content').hide();
                    }
                });

                // Schließen bei Klick außerhalb
                \$(document).on('click', function(e) {
                    if (!\$(e.target).closest('.slgs-dropdown-menu').length) {
                        \$('.slgs-dropdown-menu').removeClass('slgs-open').find('.slgs-dropdown-content').hide();
                    }
                });


                // NEU: HASH-FRAGMENT BEHANDLUNG FÜR EXTERN VERLINKTE TABS
                function checkHashForTab() {
                    var hash = window.location.hash;
                    if (hash && hash.startsWith('#slgs-tab-')) {
                        var tab_key = hash.replace('#slgs-tab-', '');
                        if (\$('#slgs-tab-' + tab_key).length) {
                             // Timeout, um sicherzustellen, dass die Seite geladen ist
                            setTimeout(function() {
                                switchTab(tab_key);
                                // Optional: Scroll zum Hub-Element
                                \$('html, body').animate({
                                    scrollTop: \$('.slgs-main-hub').offset().top - 50
                                }, 500);
                            }, 100);
                        }
                    }
                }

                // Führe beim Laden der Seite aus
                checkHashForTab();



                // --- 4.1 DETAILSEITE TAB SWITCHING LOGIK ---
                \$('body').on('click', '.slgs-content-tab-button', function() {
                    var tab_key = \$(this).data('content-tab');
                    var \$container = \$(this).closest('.slgs-content-tabs-container');
                    \$container.find('.slgs-content-tab-button').removeClass('slgs-content-tab-active');
                    \$(this).addClass('slgs-content-tab-active');
                    \$container.find('.slgs-content-tab-panel').hide();
                    \$('#slgs-content-tab-' + tab_key).fadeIn(300);

                    // FIX: Explizite Scroll-Korrektur entfernt, da sie den Fehler 'Jump to Footer' verursachte.
                    // Die CSS-Regeln sollten jetzt ausreichen, um das Scrollen zu verhindern.
                });

                // --- 5. RATING LOGIK ---
                \$('body').on('mouseenter', '.slgs-rating-stars-interactive .slgs-star-interactive', function() {
                    updateRatingStars(\$(this).parent(), \$(this).data('rating'));
                }).on('mouseleave', '.slgs-rating-stars-interactive', function() {
                    updateRatingStars(\$(this), parseInt(\$(this).data('current-rating') || 0));
                });

                \$('body').on('click', '.slgs-rating-stars-interactive .slgs-star-interactive', function() {
                    var rating = \$(this).data('rating');
                    var \$container = \$(this).parent();
                    var post_id = \$container.data('post-id');
                    var \$message = \$('#slgs-rating-message');
                    if (!slgs_ajax.is_logged_in) { return; }

                    \$message.removeClass('success error').css('color', '#66c0f4').text('Bewertung wird gesendet...');
                    var oldRating = parseInt(\$container.data('current-rating') || 0);
                    \$container.data('current-rating', rating);
                    updateRatingStars(\$container, rating);

                    \$.ajax({
                        url: slgs_ajax.ajax_url, type: 'POST',
                        data: { action: 'slgs_submit_rating', post_id: post_id, rating: rating, nonce: slgs_ajax.rating_nonce },
                        success: function(response) {
                            if (response.success) {
                                \$message.css('color', '#57cb34').text(response.data.message).addClass('success');
                                \$container.data('current-rating', response.data.rating);
                                \$('#slgs-avg-value').text(response.data.average + ' / 5.0');
                                \$('#slgs-total-votes').text('(' + response.data.total + ' Stimmen)');
                                var actionText = response.data.message.includes('aktualisiert') ? 'aktualisiert' : 'bewertet';
                                \$('#slgs-user-rating-status').html('Sie haben dieses Spiel mit <span class=\"slgs-user-rated-value\">' + response.data.rating + '</span> Sternen ' + actionText + '.');
                                updateRatingStars(\$('#slgs-avg-rating-display'), response.data.average);

                                // NEU: Update Farbstil der Rating Box
                                var avgRating = parseFloat(response.data.average.replace(',', '.'));
                                var ratingClass = (avgRating >= 4.0) ? 'slgs-rating-excellent' : (avgRating >= 3.0) ? 'slgs-rating-good' : (avgRating >= 2.0) ? 'slgs-rating-mixed' : (avgRating >= 1.0) ? 'slgs-rating-bad' : 'slgs-rating-none';
                                \$('.slgs-rating-box').removeClass('slgs-rating-excellent slgs-rating-good slgs-rating-mixed slgs-rating-bad slgs-rating-none').addClass(ratingClass);

                            } else {
                                \$message.css('color', '#ff5757').text('Fehler: ' + response.data.message);
                                \$container.data('current-rating', oldRating); updateRatingStars(\$container, oldRating);
                            }
                        },
                        error: function() {
                            \$message.css('color', '#ff5757').text('Es ist ein unbekannter Fehler aufgetreten.');
                            \$container.data('current-rating', oldRating); updateRatingStars(\$container, oldRating);
                        }
                    });
                });

                // --- 6. TEILEN LOGIK ---
                \$('body').on('click', '#slgs-share-button', function() {
                    var \$button = \$(this); var \$textSpan = \$button.find('.slgs-share-text');
                    var originalText = \$textSpan.text(); var shareUrl = window.location.href;
                    \$button.prop('disabled', true);

                    if (navigator.clipboard && navigator.clipboard.writeText) {
                        navigator.clipboard.writeText(shareUrl).then(function() {
                            \$textSpan.text('Link kopiert!');
                            setTimeout(function() { \$textSpan.text(originalText); \$button.prop('disabled', false); }, 2000);
                        }).catch(function() {
                            try {
                                var \$tempInput = \$('<input>'); \$('body').append(\$tempInput);
                                \$tempInput.val(shareUrl).select(); document.execCommand('copy'); \$tempInput.remove();
                                \$textSpan.text('Link kopiert! (Fallback)');
                                setTimeout(function() { \$textSpan.text(originalText); \$button.prop('disabled', false); }, 2000);
                            } catch (err) {
                                \$textSpan.text('Fehler beim Kopieren.');
                                setTimeout(function() { \$textSpan.text(originalText); \$button.prop('disabled', false); }, 2000);
                            }
                        });
                    }
                });

                // --- 7. GALERIE LOGIK (Einzelseite) ---
                \$('body').on('click', '.slgs-media-thumb', function() {
                    var \$thumb = \$(this);
                    var fullUrl = \$thumb.data('full-url');
                    var \$mainImage = \$('#slgs-main-screenshot');

                    if (!\$thumb.hasClass('slgs-thumb-active')) {
                        // Thumbnail aktiv setzen
                        \$('.slgs-media-thumb').removeClass('slgs-thumb-active');
                        \$thumb.addClass('slgs-thumb-active');

                        // Hauptbild wechseln (mit leichtem Fade-Effekt)
                        \$mainImage.fadeOut(150, function() {
                            \$(this).attr('src', fullUrl).fadeIn(150);
                        });
                    }
                });

                // --- 8. CHECKOUT INITIALISIERUNG ---
                var \$checkoutButton = \$('#slgs-checkout-button');
                if (\$checkoutButton.length) {
                    var initialRawTotal = parseFloat(\$checkoutButton.data('total-raw'));
                    var hasItems = \$('.slgs-cart-item').length > 0;
                    if (!isNaN(initialRawTotal)) {
                         // Nur initialisieren, wenn es Artikel gibt
                         updateCartTotal(0);
                    }
                    if (!hasItems) {
                         // Zeigt den leeren Warenkorb Zustand an
                         updateCartTotal(-10000000);
                    }
                }

            });
        ";
        wp_add_inline_script( 'slgs-frontend-logic', $inline_script );
    }
}
add_action( 'wp_enqueue_scripts', 'slgs_enqueue_styles' );
  ?>